<?php
require_once 'conexao.php';

// Busca apenas as últimas 5 vendas para o gráfico (Zoom no recente)
$query = "SELECT preco, dataehora FROM vendas ORDER BY dataehora DESC LIMIT 5";

try {
    $result = $pdo->query($query);
    $rows = $result->fetchAll(PDO::FETCH_ASSOC);

    // Se não tiver dados suficientes, cria fake para não quebrar
    if (count($rows) < 2) {
        $rows = [
            ['preco' => 0.10, 'dataehora' => date('Y-m-d H:i:s', strtotime('-1 hour'))],
            ['preco' => 0.10, 'dataehora' => date('Y-m-d H:i:s')]
        ];
    }

    // Inverte para cronológico (Antigo -> Novo)
    $rows = array_reverse($rows);

    $labels = [];
    $data = [];
    
    foreach ($rows as $row) {
        $labels[] = date('d/m H:i', strtotime($row['dataehora']));
        $data[] = (float)$row['preco'];
    }
    
    $ultimoPreco = end($data);
    
    // Calcula variação
    $penultimoPreco = isset($data[count($data)-2]) ? $data[count($data)-2] : $ultimoPreco;
    $variacao = ($penultimoPreco > 0) ? (($ultimoPreco - $penultimoPreco) / $penultimoPreco) * 100 : 0;
    
    $corTendencia = $variacao >= 0 ? '#00ff7f' : '#ff2222';
    $iconeTendencia = $variacao >= 0 ? '▲' : '▼';
    $sinal = $variacao >= 0 ? '+' : '';

} catch (PDOException $e) {
    echo "Erro no gráfico.";
    exit;
}
?>

<div class="mini-chart-wrapper">
    <div class="chart-header">
        <span class="chart-title">Preço do NFT</span>
        <div class="chart-value" style="color: <?= $corTendencia ?>">
            R$ <?= number_format($ultimoPreco, 2, ',', '.') ?>
            <span class="chart-change">
                <?= $iconeTendencia ?> <?= $sinal . number_format($variacao, 2) ?>%
            </span>
        </div>
    </div>
    
    <div class="canvas-container">
        <canvas id="nftMiniChart"></canvas>
    </div>
</div>

<script>
    if (typeof Chart === 'undefined') {
        document.write('<script src="https://cdn.jsdelivr.net/npm/chart.js"><\/script>');
    }
</script>

<style>
    .mini-chart-wrapper {
        background: rgba(0, 5, 20, 0.6); /* Fundo levemente escuro */
        border: 1px solid var(--primary);
        border-radius: 10px;
        padding: 15px;
        width: 100%;
        box-sizing: border-box;
        display: flex;
        flex-direction: column;
    }

    .chart-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 10px;
        padding-bottom: 5px;
        border-bottom: 1px dashed #333;
    }

    .chart-title {
        color: #ccc;
        font-size: 0.9rem;
        text-transform: uppercase;
        letter-spacing: 1px;
        font-weight: bold;
    }

    .chart-value {
        font-size: 1.3rem;
        font-weight: bold;
        text-shadow: 0 0 5px rgba(0,0,0,0.5);
    }

    .chart-change {
        font-size: 0.8rem;
        margin-left: 5px;
    }

    /* Altura controlada para não ficar gigante */
    .canvas-container {
        position: relative;
        height: 300px; 
        width: 100%;
    }
</style>

<script>
    document.addEventListener("DOMContentLoaded", function() {
        const ctx = document.getElementById('nftMiniChart').getContext('2d');
        const rawData = <?= json_encode($data) ?>;

        // Cria gradiente Neon Vertical
        const gradient = ctx.createLinearGradient(0, 0, 0, 300);
        gradient.addColorStop(0, 'rgba(0, 243, 255, 0.4)'); 
        gradient.addColorStop(1, 'rgba(0, 243, 255, 0.0)'); 

        new Chart(ctx, {
            type: 'line',
            data: {
                labels: <?= json_encode($labels) ?>,
                datasets: [{
                    label: 'Preço (R$)',
                    data: rawData,
                    borderColor: '#00f3ff', // Linha Azul Neon
                    borderWidth: 2,
                    backgroundColor: gradient,
                    fill: true,
                    pointBackgroundColor: '#000',
                    pointBorderColor: '#00ff7f', // Pontos Verdes
                    pointRadius: 5, // Pontos visíveis
                    pointHoverRadius: 8,
                    pointBorderWidth: 2,
                    tension: 0 // LINHAS RETAS (Sem curva)
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { display: false },
                    tooltip: {
                        backgroundColor: 'rgba(0, 0, 0, 0.9)',
                        titleColor: '#00f3ff',
                        bodyColor: '#fff',
                        borderColor: '#00f3ff',
                        borderWidth: 1,
                        displayColors: false,
                        callbacks: {
                            label: function(context) {
                                let label = 'R$ ' + context.parsed.y.toFixed(2);
                                return label;
                            },
                            afterLabel: function(context) {
                                // Lógica para mostrar a % de variação no tooltip
                                if (context.dataIndex > 0) {
                                    let prev = context.dataset.data[context.dataIndex - 1];
                                    let current = context.parsed.y;
                                    let change = ((current - prev) / prev) * 100;
                                    let sign = change >= 0 ? '+' : '';
                                    return 'Variação: ' + sign + change.toFixed(2) + '%';
                                }
                                return 'Início';
                            }
                        }
                    }
                },
                scales: {
                    x: { 
                        display: true, // Mostra datas
                        grid: { color: 'rgba(255,255,255,0.1)' },
                        ticks: { color: '#888', font: { size: 10 } }
                    }, 
                    y: { 
                        display: true, // Mostra valores
                        grid: { color: 'rgba(255,255,255,0.1)' },
                        ticks: { color: '#888' }
                    }  
                },
                layout: { padding: 5 }
            }
        });
    });
</script>