<?php
session_start();
require 'conexao.php';

// Verifica sessão básica
if (!isset($_SESSION['personagem'], $_SESSION['user_name'])) {
    die('<div class="error-msg">Sessão expirada.</div>');
}

$id_personagem = $_SESSION['personagem'];
$nome_jogador_atual = $_SESSION['personagem_nome']; // Nome do seu personagem

// 1. DESCOBRIR ONDE ESTOU E QUANDO CHEGUEI (Lógica Temporal)
// Busca o último movimento deste personagem para o local atual
$stmt = $pdo->prepare("SELECT destino, data FROM movpersonagens 
                       WHERE personagem = :id 
                       ORDER BY data DESC LIMIT 1");
$stmt->execute([':id' => $id_personagem]);
$ultimo_movimento = $stmt->fetch(PDO::FETCH_ASSOC);

// Define o local atual
$local_atual = $_SESSION['local']; // Pega da sessão para ser rápido

// Define a data de chegada. 
// Se não tiver histórico (nasceu ali), define como agora (não vê passado) ou 
// uma data antiga se preferir. Vamos usar a lógica "Só vê a partir da chegada".
if ($ultimo_movimento && $ultimo_movimento['destino'] == $local_atual) {
    $data_chegada = $ultimo_movimento['data'];
} else {
    // Se não tem movimento registrado para este local (ex: acabou de criar), 
    // assume que chegou agora ou pega as ultimas do sistema. 
    // Para seguir sua regra estrita:
    $data_chegada = date('Y-m-d H:i:s', strtotime('-1 minute')); // Dá 1 minuto de tolerância
}

// 2. CONSULTA DO CHAT (Filtrada por Local E Tempo)
$last_id = isset($_GET['last_id']) ? (int)$_GET['last_id'] : 0;

$sql = "SELECT id, personagem, fala, direcionada, dataehora 
        FROM chatpersonagens 
        WHERE local = :local 
        AND dataehora >= :chegada"; // <--- AQUI ESTÁ A MÁGICA DO TEMPO

if ($last_id > 0) {
    $sql .= " AND id > :last_id";
}

$sql .= " ORDER BY dataehora ASC";

$stmt = $pdo->prepare($sql);
$stmt->bindValue(':local', $local_atual);
$stmt->bindValue(':chegada', $data_chegada);
if ($last_id > 0) {
    $stmt->bindValue(':last_id', $last_id, PDO::PARAM_INT);
}
$stmt->execute();
$mensagens = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Calcular novo last_id para o AJAX
$new_last_id = !empty($mensagens) ? end($mensagens)['id'] : $last_id;

// --- MODO AJAX (Retorna HTML) ---
if (isset($_GET['ajax'])) {
    ob_start();
    foreach ($mensagens as $msg) {
        renderizarMensagem($msg, $nome_jogador_atual);
    }
    echo ob_get_clean();
    exit;
}

// Função para desenhar (com filtro de privacidade rigoroso)
function renderizarMensagem($msg, $meu_nome) {
    $remetente = $msg['personagem'];
    $destinatario = $msg['direcionada']; // Pode ser nome, 'Todos' ou vazio
    
    $isPropria = ($remetente === $meu_nome);
    
    // Verifica se é privada
    // Consideramos privada se tiver destinatário E não for "Todos"
    $isPrivada = (!empty($destinatario) && strtolower($destinatario) !== 'todos');

    // LÓGICA DE VISIBILIDADE (A CORREÇÃO QUE VOCÊ PEDIU)
    // Se for privada: SÓ MOSTRA se EU sou o Remetente OU EU sou o Destinatário
    if ($isPrivada) {
        if ($remetente !== $meu_nome && $destinatario !== $meu_nome) {
            return; // Pula essa mensagem, não exibe nada
        }
    }
    
    // Definição de estilos
    $classe = 'chat-msg';
    if ($isPropria) $classe .= ' msg-propria';
    else if ($isPrivada) $classe .= ' msg-privada'; // Amarelo
    else $classe .= ' msg-publica'; // Padrão
    
    $hora = date('H:i', strtotime($msg['dataehora']));
    ?>
    <div class="<?php echo $classe; ?>" data-id="<?php echo $msg['id']; ?>">
        <div class="msg-header">
            <span class="msg-author"><?php echo htmlspecialchars($remetente); ?></span>
            <?php if ($isPrivada): ?>
                <span class="msg-target" style="color:#ffd700"> ➝ <?php echo htmlspecialchars($destinatario); ?></span>
            <?php endif; ?>
            <span class="msg-time"><?php echo $hora; ?></span>
        </div>
        <div class="msg-content"><?php echo nl2br(htmlspecialchars($msg['fala'])); ?></div>
    </div>
    <?php
}
?>
<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <link rel="stylesheet" href="css/dashboardmed.css">
    <style>
        /* Estilos Específicos do Iframe */
        body {
            background: transparent;
            overflow-y: auto;
            padding: 10px;
            scrollbar-width: thin;
            scrollbar-color: var(--primary) transparent;
            font-family: 'Segoe UI', sans-serif;
        }
        .chat-msg {
            margin-bottom: 8px;
            padding: 8px 12px;
            border-radius: 6px;
            font-size: 0.9rem;
            border-left: 3px solid #555;
            background: rgba(0, 0, 0, 0.6);
            color: #eee;
            animation: fadeIn 0.3s ease;
        }
        @keyframes fadeIn { from { opacity: 0; transform: translateY(5px); } to { opacity: 1; transform: translateY(0); } }
        
        .msg-propria {
            border-left-color: var(--neon-bright);
            background: rgba(0, 0, 255, 0.15);
        }
        /* Mensagem Privada (Amarelo/Dourado) */
        .msg-privada {
            border-left-color: #ffd700;
            background: rgba(255, 215, 0, 0.15);
            border: 1px solid rgba(255, 215, 0, 0.3); /* Borda extra para destacar */
        }
        .msg-header {
            font-size: 0.8rem;
            margin-bottom: 4px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .msg-author { font-weight: bold; color: var(--primary); }
        .msg-time { font-size: 0.7rem; color: #aaa; }
        .msg-content { word-wrap: break-word; line-height: 1.4; }
        
        #new-msg-btn {
            position: fixed; bottom: 10px; right: 10px;
            background: var(--primary); color: white;
            padding: 5px 12px; border-radius: 20px; font-size: 0.8rem;
            cursor: pointer; display: none; border: 1px solid white;
            box-shadow: 0 0 10px var(--primary); z-index: 100;
        }
    </style>
</head>
<body>
    <div id="chat-content">
        <?php 
        if (empty($mensagens)) {
            echo '<div style="text-align:center; color:#888; margin-top:20px; font-style:italic; font-size:0.9rem;">O local está silencioso...<br>(Sem mensagens recentes)</div>';
        } else {
            foreach ($mensagens as $msg) {
                renderizarMensagem($msg, $nome_jogador_atual);
            }
        }
        ?>
    </div>
    <div id="new-msg-btn" onclick="scrollToBottom()">Novas mensagens ↓</div>

    <script>
        let lastId = <?php echo $new_last_id; ?>;
        let isScrolledToBottom = true;

        function scrollToBottom() {
            window.scrollTo(0, document.body.scrollHeight);
            isScrolledToBottom = true;
            document.getElementById('new-msg-btn').style.display = 'none';
        }

        window.addEventListener('scroll', () => {
            if ((window.innerHeight + window.scrollY) >= document.body.offsetHeight - 50) {
                isScrolledToBottom = true;
                document.getElementById('new-msg-btn').style.display = 'none';
            } else {
                isScrolledToBottom = false;
            }
        });

        function loadMessages() {
            fetch(`?last_id=${lastId}&ajax=1`)
                .then(r => r.text())
                .then(html => {
                    if (html.trim()) {
                        const content = document.getElementById('chat-content');
                        const temp = document.createElement('div');
                        temp.innerHTML = html;
                        const msgs = temp.querySelectorAll('.chat-msg');
                        if (msgs.length > 0) {
                            lastId = msgs[msgs.length - 1].getAttribute('data-id');
                        }
                        content.insertAdjacentHTML('beforeend', html);
                        if (isScrolledToBottom) scrollToBottom();
                        else document.getElementById('new-msg-btn').style.display = 'block';
                    }
                });
        }
        
        scrollToBottom();
        setInterval(loadMessages, 3000);
    </script>
</body>
</html>