<?php
session_start();
require '../../conex/conexao.php';

// 1. Verificação de Login
if (!isset($_SESSION['user_id'])) {
    header("Location: ../../login.php");
    exit();
}

if (isset($_POST['treinar'])) { // Ou o nome do botão que finaliza

    $personagem_id = $_SESSION['personagem'];

    try {
        // INÍCIO DA TRANSAÇÃO FINANCEIRA/NFT
        $pdo->beginTransaction();

        // 2. Busca e BLOQUEIA (FOR UPDATE)
        // Isso impede que o usuário finalize duas vezes ao mesmo tempo
        $sql = "SELECT fome, carne, soldados, situacao, paralisado FROM personagens WHERE id = :id FOR UPDATE";
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':id', $personagem_id, PDO::PARAM_INT);
        $stmt->execute();
        
        $dados = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$dados) {
            $pdo->rollBack();
            throw new Exception("Personagem não encontrado.");
        }

        // 3. Verificação Anti-Cheat (Importante!)
        // O personagem estava realmente comendo? Ou o cara tentou chamar esse arquivo direto?
        if ($dados['situacao'] != 'alimentando') {
            $pdo->rollBack();
            $_SESSION['msg'] = "Ação inválida. Você não iniciou a alimentação.";
            header("Location: ../../dashboardmed.php");
            exit();
        }

        // (Opcional) Verificar se o tempo já passou
        // if (strtotime($dados['paralisado']) > time()) { ... erro: ainda falta tempo ... }

        $fome_atual = (float)$dados['fome'];
        $estoque_carne = (int)$dados['carne'];
        $qtd_soldados = (int)$dados['soldados'];

        // Se por algum motivo ele gastou a carne em outra aba enquanto esperava
        if ($estoque_carne <= 0) {
            // Libera o personagem mas não enche a barriga
            $sqlLiberar = "UPDATE personagens SET situacao = 'livre' WHERE id = :id";
            $stmtLib = $pdo->prepare($sqlLiberar);
            $stmtLib->bindParam(':id', $personagem_id, PDO::PARAM_INT);
            $stmtLib->execute();
            
            $pdo->commit();
            $_SESSION['msg'] = "A comida acabou antes de você conseguir alimentar os soldados!";
            header("Location: ../../dashboardmed.php");
            exit();
        }

        // --- CÁLCULO DA ALIMENTAÇÃO (Lógica Robusta) ---

        $porcentagem_faltante = 100 - $fome_atual;
        
        // Se já estiver cheio (caso raro se passou na validação do início, mas possível)
        if ($porcentagem_faltante <= 0) {
            $carne_necessaria_total = 0;
        } else {
            // Regra: Carne Necessária = (Fome que falta / 100) * Soldados
            $carne_necessaria_total = ceil(($porcentagem_faltante / 100) * $qtd_soldados);
        }

        $carne_consumida = 0;
        $fome_final = 0;
        $msg_tipo = "";

        // CASO 1: Alimentação Total
        if ($estoque_carne >= $carne_necessaria_total) {
            $carne_consumida = $carne_necessaria_total;
            $fome_final = 100;
            $msg_tipo = "total";
        } 
        // CASO 2: Alimentação Parcial
        else {
            $carne_consumida = $estoque_carne; // Usa tudo que tem
            
            // Regra inversa: Quanto essa carne rende?
            $porcentagem_recuperada = ($carne_consumida / $qtd_soldados) * 100;
            $fome_final = $fome_atual + $porcentagem_recuperada;
            
            if ($fome_final > 100) $fome_final = 100;
            $msg_tipo = "parcial";
        }

        // --- APLICAÇÃO NO BANCO ---

        $situacao_nova = "livre";
        $novo_estoque = $estoque_carne - $carne_consumida;
        $fome_final_db = number_format($fome_final, 2, '.', '');

        $sqlUpdate = "UPDATE personagens SET 
                        situacao = :situacao, 
                        fome = :fome, 
                        carne = :carne 
                      WHERE id = :id";
        
        $stmtUp = $pdo->prepare($sqlUpdate);
        $stmtUp->bindParam(':situacao', $situacao_nova, PDO::PARAM_STR);
        $stmtUp->bindParam(':fome', $fome_final_db, PDO::PARAM_STR);
        $stmtUp->bindParam(':carne', $novo_estoque, PDO::PARAM_INT);
        $stmtUp->bindParam(':id', $personagem_id, PDO::PARAM_INT);

        if ($stmtUp->execute()) {
            $pdo->commit(); // Confirma a transação

            // Atualiza Sessão
            $_SESSION['fome'] = $fome_final_db;
            $_SESSION['carne'] = $novo_estoque;

            // Mensagens
            if ($msg_tipo == "total") {
                $_SESSION['msg'] = "Alimentação concluída com sucesso! Consumido: $carne_consumida carnes. Seus soldados estão totalmente alimentados.";
            } else {
                $recuperado = number_format(($fome_final - $fome_atual), 2, ',', '.');
                $_SESSION['msg'] = "Alimentação parcial. Estoque esgotado de $carne_consumida carnes, recuperou $recuperado% da fome dos seus soldados.";
            }

            header("Location: ../../dashboardmed.php");
            exit();
        } else {
            throw new Exception("Falha ao atualizar dados.");
        }

    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        $_SESSION['msg'] = "Erro ao finalizar alimentação: " . $e->getMessage();
        header("Location: ../../dashboardmed.php");
        exit();
    }
}
?>