<?php
session_start();
require '../conexao.php';

// 1. VERIFICAÇÕES DE SEGURANÇA INICIAIS
if (!isset($_SESSION['user_id'])) {
    header("Location: ../login.php");
    exit();
}

$user_id = $_SESSION['user_id']; 
$user_name = $_SESSION['user_name']; 

if ($_SERVER['REQUEST_METHOD'] !== 'POST' || !isset($_POST['vender'])) {
    header('Location: mercado.php');
    exit;
}

// Validação CSRF
if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
    $_SESSION['mensagem'] = 'Erro de segurança: Token inválido. Tente novamente.';
    header('Location: mercado.php');
    exit;
}

// 2. VERIFICAÇÃO DE PIN
$ip = $_SERVER['REMOTE_ADDR'];
$stmt = $pdo->prepare("SELECT COUNT(*) FROM tentativas_pin WHERE jogador = ? AND timestamp >= NOW() - INTERVAL 24 HOUR");
$stmt->execute([$user_name]);
if ($stmt->fetchColumn() >= 7) {
    $_SESSION['mensagem'] = "PIN BLOQUEADO: Excesso de tentativas. Aguarde 24h.";
    header('Location: mercado.php');
    exit;
}

$pin_digitado = $_POST['pin'] ?? '';
if (!password_verify($pin_digitado, $_SESSION['user_pin'])) {
    $stmt = $pdo->prepare("INSERT INTO tentativas_pin (ip, jogador, local, timestamp) VALUES (?, ?, 'venda_personagem', NOW())");
    $stmt->execute([$ip, $user_name]);
    $_SESSION['mensagem'] = 'PIN Incorreto.';
    header('Location: mercado.php');
    exit;
}

// 3. VALIDAÇÃO DOS DADOS
$personagem_id = filter_input(INPUT_POST, 'personagem', FILTER_VALIDATE_INT);
$valor_raw = trim($_POST['valor'] ?? '');
$valor_formatado = str_replace(',', '.', $valor_raw);
$valor_venda_pedido = filter_var($valor_formatado, FILTER_VALIDATE_FLOAT);

if (!$personagem_id || !$valor_venda_pedido || $valor_venda_pedido <= 0) {
    $_SESSION['mensagem'] = 'Dados inválidos.';
    header('Location: mercado.php');
    exit;
}

// 4. VERIFICAR PROPRIEDADE
$stmt = $pdo->prepare("SELECT * FROM personagens WHERE id = :id AND jogador = :jogador LIMIT 1");
$stmt->execute([':id' => $personagem_id, ':jogador' => $user_name]);
$personagem = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$personagem || $personagem['situacao'] == 'vendendo') {
    $_SESSION['mensagem'] = 'Personagem indisponível para venda.';
    header('Location: mercado.php');
    exit;
}

$nome_personagem = $personagem['personagem'];

// ==================================================================================
// LÓGICA CENTRAL: MATCHING (VENDA AUTOMÁTICA)
// ==================================================================================

$stmtOrdem = $pdo->prepare("SELECT * FROM ordens_compra WHERE valor >= :valor AND user != :user_name ORDER BY valor DESC, dataehora ASC LIMIT 1");
$stmtOrdem->execute([':valor' => $valor_venda_pedido, ':user_name' => $user_name]);
$ordemCompra = $stmtOrdem->fetch(PDO::FETCH_ASSOC);

$mensagem_resultado = "";
$tipo_resultado = ""; 
$detalhes_transacao = ""; 

if ($ordemCompra) {
    // --- VENDA AUTOMÁTICA ---
    $id_ordem_compra = $ordemCompra['id'];
    $comprador = $ordemCompra['user'];
    $valor_final = $ordemCompra['valor']; 
    
    try {
        $pdo->beginTransaction();

        // 1. Transferir Personagem
        $stmtTransf = $pdo->prepare("UPDATE personagens SET jogador = :comprador, situacao = NULL, datamovimentacao = NOW() WHERE id = :id");
        $stmtTransf->execute([':comprador' => $comprador, ':id' => $personagem_id]);

        // 2. Atualizar Ordem de Compra
        if ($ordemCompra['quantidade'] > 1) {
            $stmtUpd = $pdo->prepare("UPDATE ordens_compra SET quantidade = quantidade - 1 WHERE id = :id");
            $stmtUpd->execute([':id' => $id_ordem_compra]);
        } else {
            $stmtDel = $pdo->prepare("DELETE FROM ordens_compra WHERE id = :id");
            $stmtDel->execute([':id' => $id_ordem_compra]);
        }

        // 3. Registrar Financeiro
        $data_hora = date('Y-m-d H:i:s');
        $ref = "Venda Automática: $nome_personagem (Ordem Compra #$id_ordem_compra)";
        
        $stmtW = $pdo->prepare("SELECT wallet FROM usersblock WHERE user = ?");
        $stmtW->execute([$comprador]);
        $wallet_comprador = $stmtW->fetchColumn();

        $hash = hash('sha256', $user_name . $comprador . $valor_final . microtime());
        $link = substr(hash('sha1', $hash), 0, 40);
        
        $stmtMem = $pdo->prepare("INSERT INTO mempool (remetente, destinatario, valor, dataehora, referencia, hash, link) VALUES (:rem, :dest, :val, :data, :ref, :hash, :link)");
        $stmtMem->execute([
            ':rem' => $wallet_comprador,
            ':dest' => $_SESSION['user_wallet'],
            ':val' => $valor_final,
            ':data' => $data_hora,
            ':ref' => $ref,
            ':hash' => $hash,
            ':link' => $link
        ]);
        
        $id_transacao = $pdo->lastInsertId();

        // 4. Registrar Venda (Gráfico)
        $stmtGraf = $pdo->prepare("INSERT INTO vendas (quantidade, referencia, preco, dataehora, vendedor, comprador, mediacao) VALUES (1, :ref, :preco, :data, :vend, :comp, 'Automático')");
        $stmtGraf->execute([':ref' => $ref, ':preco' => $valor_final, ':data' => $data_hora, ':vend' => $user_name, ':comp' => $comprador]);

        // 5. Logs
        $log = "Venda Auto: $nome_personagem de $user_name p/ $comprador por R$ $valor_final (Ordem #$id_ordem_compra)";
        $stmtLog = $pdo->prepare("INSERT INTO infossistema (informacao, p1, p2, origem, dataehora) VALUES (:info, :p1, :p2, 'Mercado Auto', :data)");
        $stmtLog->execute([':info' => $log, ':p1' => $user_name, ':p2' => $comprador, ':data' => $data_hora]);

        // 6. Notificação
        $notif = "Sua ordem de compra #$id_ordem_compra foi executada! Você comprou $nome_personagem.";
        $stmtNotif = $pdo->prepare("INSERT INTO notificacoes_sis (tipo, notificacao, user1, user2, dataehora) VALUES ('compra', :msg, 'Sistema', :dest, :data)");
        $stmtNotif->execute([':msg' => $notif, ':dest' => $comprador, ':data' => $data_hora]);

        $pdo->commit();
        
        // --- RESULTADO: MENSAGEM EXPLICATIVA ---
        $tipo_resultado = "sucesso";
        $mensagem_resultado = "Venda Instantânea Realizada!";
        
        $detalhes_transacao = "
            <div style='text-align:left; background:rgba(0,0,0,0.5); padding:20px; border-radius:8px; margin-top:15px; border:1px dashed #28a745; font-size: 0.95rem; line-height: 1.5;'>
                <p style='color:#fff; margin-bottom:15px;'>
                    <i class='fas fa-info-circle' style='color:#00f3ff'></i> 
                    Encontramos uma <strong>Ordem de Compra</strong> no mercado compatível com o valor que você pediu. Por isso, a venda foi executada imediatamente!
                </p>
                <hr style='border-color:#333; margin: 10px 0;'>
                <p><strong>Personagem Vendido:</strong> <span style='color:#00f3ff'>$nome_personagem</span></p>
                <p><strong>Valor Pedido por você:</strong> R$ " . number_format($valor_venda_pedido, 2, ',', '.') . "</p>
                <p><strong>Valor Pago (Ordem #$id_ordem_compra):</strong> <span style='color:#28a745; font-weight:bold;'>R$ " . number_format($valor_final, 2, ',', '.') . "</span></p>
                <hr style='border-color:#333; margin: 10px 0;'>
                <p style='font-size:0.8rem; color:#aaa;'>
                    ID Transação (Mempool): #$id_transacao<br>
                    Data: " . date('d/m/Y H:i:s', strtotime($data_hora)) . "
                </p>
                <div style='text-align:center; margin-top:15px; color:#28a745; font-weight:bold; background:rgba(40,167,69,0.1); padding:10px; border-radius:5px;'>
                    O valor já foi creditado em seu saldo.
                </div>
            </div>";

    } catch (Exception $e) {
        if ($pdo->inTransaction()) $pdo->rollBack();
        $tipo_resultado = "erro";
        $mensagem_resultado = "Erro técnico na transação: " . $e->getMessage();
    }

} else {
    // --- CENÁRIO B: CRIAR ANÚNCIO ---
    $stmtCheck = $pdo->prepare("SELECT id FROM ordem_vendas WHERE personagem = ?");
    $stmtCheck->execute([$nome_personagem]);
    
    if ($stmtCheck->rowCount() > 0) {
        $tipo_resultado = "erro";
        $mensagem_resultado = "Este personagem já está anunciado.";
    } else {
        try {
            $pdo->beginTransaction();
            
            $stmtIns = $pdo->prepare("INSERT INTO ordem_vendas (jogador, personagem, valor, dataehora) VALUES (:jog, :pers, :val, NOW())");
            $stmtIns->execute([':jog' => $user_name, ':pers' => $nome_personagem, ':val' => $valor_venda_pedido]);
            
            $stmtUpd = $pdo->prepare("UPDATE personagens SET situacao = 'vendendo' WHERE id = :id");
            $stmtUpd->execute([':id' => $personagem_id]);
            
            $pdo->commit();
            
            $tipo_resultado = "anuncio";
            $mensagem_resultado = "Anúncio Criado com Sucesso!";
            $detalhes_transacao = "
                <div style='text-align:center; padding:15px; color:#ddd;'>
                    <p>Não havia ordens de compra compatíveis no momento.</p>
                    <p>O personagem <strong>$nome_personagem</strong> foi listado no mercado por <strong style='color:#28a745'>R$ " . number_format($valor_venda_pedido, 2, ',', '.') . "</strong>.</p>
                    <p style='font-size:0.9rem; color:#aaa; margin-top:10px;'>Aguarde um comprador interessado.</p>
                </div>";
            
        } catch (Exception $e) {
            if ($pdo->inTransaction()) $pdo->rollBack();
            $tipo_resultado = "erro";
            $mensagem_resultado = "Erro ao criar anúncio: " . $e->getMessage();
        }
    }
}
?>

<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Resultado da Venda - Mavilla</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="../css/dashboard.css">
    <style>
        body {
            background: url('../rpg-login-background.jpg') no-repeat center center fixed;
            background-size: cover;
            display: flex; align-items: center; justify-content: center; min-height: 100vh;
        }
        .result-card {
            background-color: rgba(0, 5, 30, 0.95);
            border: 2px solid var(--primary);
            box-shadow: 0 0 30px rgba(0,0,255,0.5);
            border-radius: 15px;
            padding: 30px;
            width: 90%; max-width: 600px;
            text-align: center;
        }
        .icon-result { font-size: 3.5rem; margin-bottom: 15px; }
        
        .success-color { color: #28a745; text-shadow: 0 0 10px #28a745; }
        .info-color { color: #00f3ff; text-shadow: 0 0 10px #00f3ff; }
        .error-color { color: #ff2222; text-shadow: 0 0 10px #ff2222; }

        h2 { text-transform: uppercase; margin-bottom: 15px; font-size: 1.5rem; }
        
        .btn-back {
            display: inline-block;
            background: #000; border: 2px solid var(--primary); color: #fff;
            padding: 10px 25px; border-radius: 6px; font-weight: bold;
            text-transform: uppercase; text-decoration: none; margin-top: 20px;
            transition: 0.3s;
        }
        .btn-back:hover { background: var(--primary); color: #000; box-shadow: 0 0 15px var(--neon-bright); }
    </style>
</head>
<body>

    <div class="result-card" style="border-color: <?php echo ($tipo_resultado == 'erro') ? '#ff2222' : (($tipo_resultado == 'sucesso') ? '#28a745' : '#00f3ff'); ?>">
        
        <?php if ($tipo_resultado == 'sucesso'): ?>
            <div class="icon-result success-color">✔</div>
            <h2 class="success-color"><?= $mensagem_resultado ?></h2>
            <?= $detalhes_transacao ?>
            
        <?php elseif ($tipo_resultado == 'anuncio'): ?>
            <div class="icon-result info-color">📢</div>
            <h2 class="info-color"><?= $mensagem_resultado ?></h2>
            <?= $detalhes_transacao ?>
            
        <?php else: ?>
            <div class="icon-result error-color">✖</div>
            <h2 class="error-color">Ops! Algo deu errado.</h2>
            <p style="color:#ddd; font-size:1.1rem;"><?= $mensagem_resultado ?></p>
        <?php endif; ?>

        <a href="mercado.php" class="btn-back">Voltar ao Mercado</a>
        
        <?php if ($tipo_resultado == 'anuncio'): ?>
            <a href="ordens_venda.php" class="btn-back" style="margin-left:10px; border-color:#ffd700; color:#ffd700;">Meus Anúncios</a>
        <?php endif; ?>
        
    </div>

</body>
</html>