<?php
session_start();
require '../conexao.php';

if (!isset($_SESSION['user_id'])) {
    header("Location: ../login.php");
    exit();
}

// CSRF Token
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

$user_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name'];
$user_wallet = $_SESSION['user_wallet'];

// --- PROCESSAMENTO DA VENDA ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['venderFichas'])) {

    // 1. Validações de Segurança
    if ($_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        die("Erro de Segurança: Token Inválido.");
    }

    // Verificação de PIN
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM tentativas_pin WHERE jogador = ? AND timestamp >= NOW() - INTERVAL 24 HOUR");
    $stmt->execute([$user_name]);
    if ($stmt->fetchColumn() >= 7) {
        $_SESSION['msg'] = "PIN BLOQUEADO. Aguarde 24h.";
        header("Location: vender_fichas.php");
        exit;
    }

    $pin_digitado = $_POST['pin'];
    if (!password_verify($pin_digitado, $_SESSION['user_pin'])) {
        $stmt = $pdo->prepare("INSERT INTO tentativas_pin (ip, jogador, local, timestamp) VALUES (?, ?, 'venda_fichas', NOW())");
        $stmt->execute([$_SERVER['REMOTE_ADDR'], $user_name]);
        $_SESSION['msg'] = 'PIN Incorreto.';
        header("Location: vender_fichas.php");
        exit;
    }

    // 2. Validar Dados
    $quantidade_venda = (int)$_POST['quantidade'];
    $valor_unitario = filter_var(str_replace(',', '.', $_POST['preco']), FILTER_VALIDATE_FLOAT);

    if ($quantidade_venda <= 0 || $valor_unitario <= 0) {
        $_SESSION['msg'] = "Quantidade ou Preço inválidos.";
        header("Location: vender_fichas.php");
        exit;
    }

    // 3. Verificar se tem fichas suficientes
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM fichaspersonagens WHERE jogador = ? AND situacao = '1'");
    $stmt->execute([$user_name]);
    $fichas_disponiveis = $stmt->fetchColumn();

    if ($fichas_disponiveis < $quantidade_venda) {
        $_SESSION['msg'] = "Você só tem $fichas_disponiveis fichas disponíveis.";
        header("Location: vender_fichas.php");
        exit;
    }

    // ==================================================================================
    // 4. BUSCA DE COMPRADORES (MATCHING)
    // ==================================================================================
    // Procura Ordens de Compra que paguem IGUAL ou MAIS do que eu quero
    $sqlOrdens = "SELECT id, quantidade, valor, user FROM ordens_compra 
                  WHERE valor >= :valor AND user != :eu 
                  ORDER BY valor DESC, dataehora ASC";
    
    $stmtOrdens = $pdo->prepare($sqlOrdens);
    $stmtOrdens->execute([':valor' => $valor_unitario, ':eu' => $user_name]);
    $ordens_compra = $stmtOrdens->fetchAll(PDO::FETCH_ASSOC);

    $quantidade_restante = $quantidade_venda;
    $total_ganho = 0;
    $detalhes_venda = [];
    $data_hora = date('Y-m-d H:i:s');

    try {
        $pdo->beginTransaction();

        // Loop de Venda Automática
        foreach ($ordens_compra as $ordem) {
            if ($quantidade_restante <= 0) break;

            $id_ordem = $ordem['id'];
            $comprador = $ordem['user'];
            $valor_pago = $ordem['valor']; // Vende pelo valor que o comprador ofereceu
            $qtd_demanda = $ordem['quantidade'];

            // Calcula quantos vender para esta ordem
            $qtd_transacionada = min($qtd_demanda, $quantidade_restante);
            
            // A. Atualizar Ordem de Compra
            if ($qtd_transacionada < $qtd_demanda) {
                $stmtUpdOrdem = $pdo->prepare("UPDATE ordens_compra SET quantidade = quantidade - :qtd WHERE id = :id");
                $stmtUpdOrdem->execute([':qtd' => $qtd_transacionada, ':id' => $id_ordem]);
            } else {
                $stmtDelOrdem = $pdo->prepare("DELETE FROM ordens_compra WHERE id = :id");
                $stmtDelOrdem->execute([':id' => $id_ordem]);
            }

            // B. Transferir Fichas (Pegar X fichas minhas e passar pro comprador)
            $stmtTransf = $pdo->prepare("UPDATE fichaspersonagens SET jogador = :novo_dono, situacao = '1', preco = 0 WHERE jogador = :eu AND situacao = '1' LIMIT :limite");
            // PDO LIMIT não aceita bind direto em algumas versões, então fazemos direto na query se for int seguro
            $sqlTransf = "UPDATE fichaspersonagens SET jogador = '$comprador', situacao = '1', preco = 0 WHERE jogador = '$user_name' AND situacao = '1' LIMIT $qtd_transacionada";
            $pdo->exec($sqlTransf);

            // C. Pagamento (Mempool)
            // Busca wallet do comprador
            $stmtW = $pdo->prepare("SELECT wallet FROM usersblock WHERE user = ?");
            $stmtW->execute([$comprador]);
            $wallet_comprador = $stmtW->fetchColumn();

            $valor_total_transacao = bcmul($valor_pago, $qtd_transacionada, 2);
            $ref = "Venda Automática de $qtd_transacionada Fichas para ordem de compra de id #$id_ordem";
            
            // Inserir na Mempool (Dinheiro sai do comprador pra mim)
            // Nota: O dinheiro já saiu da wallet do comprador quando ele criou a ordem, 
            // então aqui creditamos na do vendedor vindo do "Sistema/Escrow".
            // Mas seguindo sua lógica anterior, é transferência direta.
            
            $stmtMem = $pdo->prepare("INSERT INTO mempool (remetente, destinatario, valor, dataehora, referencia) VALUES (:rem, :dest, :val, :data, :ref)");
            $stmtMem->execute([
                ':rem' => $wallet_comprador, // Ou 'SISTEMA_RESERVA' se o dinheiro já foi debitado na criação da ordem
                ':dest' => $_SESSION['user_wallet'],
                ':val' => $valor_total_transacao,
                ':data' => $data_hora,
                ':ref' => $ref
            ]);
            
            $id_transacao = $pdo->lastInsertId();

            // D. Logs e Notificações
            $log = "Venda Auto: $qtd_transacionada Fichas de $user_name p/ $comprador por R$ $valor_pago cada";
            $stmtLog = $pdo->prepare("INSERT INTO infossistema (informacao, p1, p2, origem, dataehora) VALUES (:info, :p1, :p2, 'Mercado Fichas', :data)");
            $stmtLog->execute([':info' => $log, ':p1' => $user_name, ':p2' => $comprador, ':data' => $data_hora]);

            $notif = "Sua ordem de compra #$id_ordem comprou $qtd_transacionada fichas.";
            $stmtNotif = $pdo->prepare("INSERT INTO notificacoes_sis (tipo, notificacao, user1, user2, dataehora) VALUES ('compra', :msg, 'Sistema', :dest, :data)");
            $stmtNotif->execute([':msg' => $notif, ':dest' => $comprador, ':data' => $data_hora]);

            // Atualiza Loop
            $quantidade_restante -= $qtd_transacionada;
            $total_ganho += $valor_total_transacao;
            $detalhes_venda[] = "$qtd_transacionada fichas para $comprador (R$ " . number_format($valor_pago, 2, ',', '.') . " cada)";
        }

        // 5. SE SOBROU FICHA, CRIA ANÚNCIO DE VENDA
        if ($quantidade_restante > 0) {
            // Muda status das fichas restantes para 'vendendo' com o preço definido
            $sqlVenda = "UPDATE fichaspersonagens SET situacao = 'vendendo', preco = $valor_unitario WHERE jogador = '$user_name' AND situacao = '1' LIMIT $quantidade_restante";
            $pdo->exec($sqlVenda);
        }

        $pdo->commit();

        // === RESULTADO ===
        $qtd_vendida = $quantidade_venda - $quantidade_restante;
        $html_resumo = "<h3>Resultado da Venda</h3>";
        
        if ($qtd_vendida > 0) {
            $html_resumo .= "<p style='color:#28a745'>✅ <strong>$qtd_vendida Fichas vendidas imediatamente!</strong></p>";
            $html_resumo .= "<p>Total Recebido: <strong>R$ " . number_format($total_ganho, 2, ',', '.') . "</strong></p>";
            $html_resumo .= "<ul>";
            foreach ($detalhes_venda as $d) { $html_resumo .= "<li>$d</li>"; }
            $html_resumo .= "</ul>";
        }
        
        if ($quantidade_restante > 0) {
            $html_resumo .= "<p style='color:#00f3ff'>📢 <strong>$quantidade_restante Fichas</strong> foram listadas no mercado por R$ " . number_format($valor_unitario, 2, ',', '.') . ".</p>";
        }

        $_SESSION['resultado_html'] = $html_resumo;

    } catch (Exception $e) {
        if ($pdo->inTransaction()) $pdo->rollBack();
        $_SESSION['msg'] = "Erro: " . $e->getMessage();
    }
}

// Contagem para exibir na tela
$stmt = $pdo->prepare("SELECT COUNT(*) FROM fichaspersonagens WHERE jogador = ? AND situacao = '1'");
$stmt->execute([$user_name]);
$fichas_disponiveis = $stmt->fetchColumn();
?>

<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Vender Fichas - Mavilla</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="../css/dashboard.css">
    <style>
        body {
            background: url('../rpg-login-background.jpg') no-repeat center center fixed;
            background-size: cover;
            color: white;
            min-height: 100vh;
            display: flex; align-items: center; justify-content: center;
        }
        .neon-card {
            background-color: rgba(0, 5, 30, 0.9);
            border: 2px solid var(--primary);
            box-shadow: 0 0 20px rgba(0, 0, 255, 0.3);
            border-radius: 15px;
            padding: 30px;
            width: 90%; max-width: 500px;
            text-align: center;
        }
        
        h1 {
            color: var(--neon-bright);
            text-transform: uppercase;
            text-shadow: 0 0 10px var(--primary);
            margin-bottom: 20px;
            font-family: 'Times New Roman', serif;
            font-size: 2rem;
        }

        input {
            background-color: #000;
            border: 2px solid var(--primary);
            color: #fff;
            padding: 12px;
            border-radius: 6px;
            width: 100%;
            margin-bottom: 15px;
            text-align: center;
            font-size: 1rem;
            box-sizing: border-box;
            outline: none;
        }
        input:focus { box-shadow: 0 0 15px var(--neon-bright); border-color: #fff; }

        button {
            background-color: #000;
            color: #00ff7f; /* Verde Dinheiro */
            border: 2px solid #00ff7f;
            padding: 12px;
            border-radius: 8px;
            width: 100%;
            font-weight: bold;
            text-transform: uppercase;
            cursor: pointer;
            transition: 0.3s;
            box-shadow: 0 0 5px rgba(0, 255, 127, 0.4);
        }
        button:hover {
            background-color: #00ff7f;
            color: #000;
            box-shadow: 0 0 20px #00ff7f;
        }
        
        .btn-back {
            display: inline-block; margin-top: 15px; color: #ccc; text-decoration: none; font-size: 0.9rem;
        }
        .btn-back:hover { color: #fff; text-decoration: underline; }
        
        .result-box {
            background: rgba(255,255,255,0.1); padding: 15px; border-radius: 8px; margin-bottom: 20px; text-align: left;
        }
    </style>
</head>
<body>

    <div class="neon-card">
        
        <?php if (isset($_SESSION['resultado_html'])): ?>
            <div class="result-box">
                <?= $_SESSION['resultado_html'] ?>
            </div>
            <?php unset($_SESSION['resultado_html']); ?>
            
            <a href="../mercado/mercado.php"><button style="border-color:var(--primary); color:var(--primary);">Voltar ao Mercado</button></a>
            
        <?php else: ?>
            <h1>Vender Fichas</h1>
            
            <?php if (isset($_SESSION['msg'])): ?>
                <div style="color: #ff2222; margin-bottom: 15px; font-weight: bold;">
                    <?= $_SESSION['msg'] ?>
                </div>
                <?php unset($_SESSION['msg']); ?>
            <?php endif; ?>

            <p style="color:#ccc; margin-bottom:20px;">Disponíveis: <strong style="color:#00f3ff"><?= $fichas_disponiveis ?></strong> Fichas</p>

            <form method="POST">
                <input type="number" name="quantidade" placeholder="Quantidade" min="1" max="<?= $fichas_disponiveis ?>" required>
                <input type="text" id="moneyInput" name="preco" placeholder="Preço Unitário (R$)" required>
                <input type="password" name="pin" placeholder="PIN de Segurança" maxlength="4" required>
                
                <input type="hidden" name="csrf_token" value="<?= $_SESSION['csrf_token'] ?>">
                
                <button type="submit" name="venderFichas" onclick="return confirm('Confirmar a venda?')">Vender Fichas</button>
            </form>
            
            <a href="../mercado/mercado.php" class="btn-back">Cancelar e Voltar</a>
        <?php endif; ?>

    </div>

    <script>
        const inputMoney = document.getElementById('moneyInput');
        if(inputMoney){
            inputMoney.addEventListener('input', function(e) {
                let value = e.target.value.replace(/\D/g, '');
                if(value === "") { this.value = ""; return; }
                let formatted = (Number(value) / 100).toFixed(2);
                this.value = formatted;
            });
        }
    </script>

</body>
</html>