<?php
// transferir_personagem.php (BLINDADO, STATUS LIVRE & LOG COMPLETO)

// 1. SEGURANÇA E CONEXÃO
require '../conexao.php';
require '../config_seguranca.php';

// 2. VERIFICAÇÃO DE LOGIN
if (!isset($_SESSION['user_id'])) {
    header("Location: ../login.php");
    exit();
}

$user_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name'];
$csrf_token = $_SESSION['csrf_token'];

// Captura mensagens de feedback
$mensagem_sucesso = "";
$mensagem_erro = "";

if (isset($_SESSION['msg_sucesso'])) {
    $mensagem_sucesso = $_SESSION['msg_sucesso'];
    unset($_SESSION['msg_sucesso']);
}
if (isset($_SESSION['msg_erro'])) {
    $mensagem_erro = $_SESSION['msg_erro'];
    unset($_SESSION['msg_erro']);
}

// --- PROCESSAMENTO DA TRANSFERÊNCIA ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['transferir_personagem'])) {
    
    // 3. BARREIRA CSRF
    verificarTokenCSRF();

    $personagem_id = filter_input(INPUT_POST, 'personagem_id', FILTER_VALIDATE_INT);
    $novo_jogador_nome = trim($_POST['novo_jogador'] ?? '');
    $pin_digitado = $_POST['pin'] ?? '';
    
    try {
        // A. Validação Básica
        if (empty($pin_digitado)) throw new Exception("Digite o PIN de segurança.");
        if (empty($novo_jogador_nome)) throw new Exception("Digite o nome do destinatário.");
        if (!$personagem_id) throw new Exception("Selecione um personagem.");
        
        // Impedir transferência para si mesmo
        if (strcasecmp($novo_jogador_nome, $user_name) === 0) {
            throw new Exception("Você não pode transferir para si mesmo.");
        }

        // B. Validação do PIN (Direto do Banco)
        $stmtPin = $pdo->prepare("SELECT pin FROM usersblock WHERE id = ? LIMIT 1");
        $stmtPin->execute([$user_id]);
        $hash_real = $stmtPin->fetchColumn();
        
        if (!$hash_real || !password_verify($pin_digitado, $hash_real)) {
            sleep(1); 
            throw new Exception("PIN incorreto. Transação negada.");
        }

        // INÍCIO DA TRANSAÇÃO ATÔMICA
        $pdo->beginTransaction();

        // C. Verificar Destinatário (BUSCANDO A WALLET AGORA)
        $stmtDest = $pdo->prepare("SELECT user, wallet FROM usersblock WHERE user = ? LIMIT 1");
        $stmtDest->execute([$novo_jogador_nome]);
        $dados_destinatario = $stmtDest->fetch(PDO::FETCH_ASSOC);
        
        if (!$dados_destinatario) {
            throw new Exception("O jogador '$novo_jogador_nome' não existe.");
        }
        
        $nome_destinatario_real = $dados_destinatario['user'];
        $wallet_destinatario = $dados_destinatario['wallet']; 

        // D. Verificar Personagem com TRAVA (FOR UPDATE)
        $stmtChar = $pdo->prepare("SELECT * FROM personagens WHERE id = ? AND jogador = ? LIMIT 1 FOR UPDATE");
        $stmtChar->execute([$personagem_id, $user_name]);
        $personagem = $stmtChar->fetch(PDO::FETCH_ASSOC);

        if (!$personagem) {
            throw new Exception("Personagem não encontrado ou não pertence a você.");
        }

        // --- SEGURANÇA DE INTEGRIDADE (Limpeza de Mercado) ---
        $stmtDelVenda = $pdo->prepare("DELETE FROM ordem_vendas WHERE personagem = ? AND jogador = ?");
        $stmtDelVenda->execute([$personagem['personagem'], $user_name]);

        // --- E. LOG NO INFOSSISTEMA (PADRÃO BLOCKCHAIN COM ID) ---
        
        // 1. Buscar Wallet do Remetente (Você)
        $stmtSender = $pdo->prepare("SELECT wallet FROM usersblock WHERE id = ? LIMIT 1");
        $stmtSender->execute([$user_id]);
        $wallet_remetente = $stmtSender->fetchColumn();

        // 2. Montar o texto COM O ID DO PERSONAGEM
        // "A wallet X transferiu o personagem Nome (ID: 123) para a wallet Y"
        $texto_blockchain = "Carteira $wallet_remetente transferiu o personagem {$personagem['personagem']} (ID: {$personagem['id']}) para a carteira $wallet_destinatario";

        // 3. Inserir no Infossistema
        $stmtLog = $pdo->prepare("INSERT INTO infossistema (informacao, p1, p2, origem, dataehora) VALUES (?, ?, ?, 'Transferência', NOW())");
        // Salvamos as Wallets em P1 e P2 para indexação técnica
        $stmtLog->execute([$texto_blockchain, $wallet_remetente, $wallet_destinatario]);

        // F. Executar Transferência (Resetando status para evitar bugs)
        $stmtUpd = $pdo->prepare("UPDATE personagens SET jogador = ?, datamovimentacao = NOW(), situacao = 'livre', ativo = 0 WHERE id = ?");
        $stmtUpd->execute([$nome_destinatario_real, $personagem_id]);

        $pdo->commit();

        $_SESSION['msg_sucesso'] = "Personagem <strong>{$personagem['personagem']}</strong> enviado para <strong>$nome_destinatario_real</strong> com sucesso!";
        header("Location: " . $_SERVER['PHP_SELF']);
        exit();

    } catch (Exception $e) {
        if ($pdo->inTransaction()) $pdo->rollBack();
        $_SESSION['msg_erro'] = $e->getMessage();
        header("Location: " . $_SERVER['PHP_SELF']);
        exit();
    }
}

// G. Leitura para o Select
$stmt = $pdo->prepare("SELECT id, personagem, situacao, ativo FROM personagens WHERE jogador = ? ORDER BY personagem ASC");
$stmt->execute([$user_name]);
$personagens = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Transferir Personagens - Ferro & Fogo</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="../css/dashboard.css">
    <script>
        function confirmarTransferencia(form) {
            const select = form.personagem_id;
            const option = select.options[select.selectedIndex];
            const personagem = option.text;
            const novoJogador = form.novo_jogador.value;
            
            const status = option.getAttribute('data-status');
            
            if (!personagem || select.value === "" || !novoJogador) {
                alert('Por favor, preencha todos os campos!');
                return false;
            }
            
            let avisoExtra = "";
            if(status !== 'livre' && status !== '1') {
                avisoExtra = "\n⚠️ AVISO: Este personagem não está 'Livre'. Ao transferir, ele será removido do mercado/batalha e resetado para o novo dono.\n";
            }
            
            return confirm(`ATENÇÃO EXTREMA: CONFIRMAR TRANSFERÊNCIA\n${avisoExtra}\nVocê está prestes a transferir:\n>> ${personagem} <<\n\nPara o jogador:\n>> ${novoJogador} <<\n\nEsta ação é IRREVERSÍVEL!\nDeseja continuar?`);
        }
    </script>
</head>
<body>
    <div class="light-rays"></div>

    <div class="container"> 
        
        <div class="card" style="margin-top: 30px; margin-bottom: 50px;">
            
            <div class="listing-header-centered">
                <h1 style="color: var(--primary); font-family: 'Times New Roman', serif; text-transform: uppercase; letter-spacing: 2px; text-shadow: var(--text-neon-glow); margin-bottom: 25px; font-size: 2rem;">
                    RPG ERA MEDIEVAL: Ferro & Fogo Pelo Poder
                </h1>
                
                <div class="btn-group btn-container-center" style="margin-bottom: 25px;"><center>
                    <a href="../admpersonagens.php">
                        <button style="margin:0; padding: 12px 50px; width: auto;">Administração de Fichas & Personagens</button>
                    </a><br>
                    <a href="../dashboard.php">
                        <button style="margin:0; padding: 12px 50px; width: auto;">Menu Principal</button>
                    </a>
                    </center>
                </div>

                <h2 style="color: white; text-transform: uppercase; letter-spacing: 1px; margin-bottom: 15px;">
                    Transferir Personagens
                </h2>

                <p class="listing-intro">
                    Envie um de seus personagens para outro jogador.<br>
                    <small style="color:#ffcc00;">Você pode transferir personagens em qualquer situação (Ativos, Vendendo, etc).</small>
                </p>
            </div>
            
            <?php if ($mensagem_sucesso): ?>
                <div class="message-neon success" style="border: 1px solid #00ff00; background: rgba(0,255,0,0.1); color: #00ff00; padding: 15px; margin: 10px auto; max-width: 600px; text-align: center; border-radius: 8px;">
                    <?= $mensagem_sucesso ?>
                </div>
            <?php endif; ?>
            
            <?php if ($mensagem_erro): ?>
                <div class="message-neon error" style="border: 1px solid #ff2222; background: rgba(255,0,0,0.1); color: #ff2222; padding: 15px; margin: 10px auto; max-width: 600px; text-align: center; border-radius: 8px;">
                    <?= htmlspecialchars($mensagem_erro) ?>
                </div>
            <?php endif; ?>

            <?php if (empty($personagens)): ?>
                <div class="sem-personagens-neon" style="text-align: center; padding: 30px; color: #ccc; font-size: 1.2rem;">
                    <p>Você não possui personagens.</p>
                </div>
            <?php else: ?>
                <div class="form-stack-container">
                    <form method="POST" onsubmit="return confirmarTransferencia(this)">
                        <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                        
                        <div>
                            <label for="personagem_id">Selecione o Personagem:</label>
                            <select name="personagem_id" id="personagem_id" required class="form-select-neon">
                                <option value="">-- Selecione --</option>
                                <?php foreach ($personagens as $p): 
                                    $statusLabel = ($p['situacao'] == 'livre' || $p['situacao'] == '1') ? '' : ' (' . ucfirst($p['situacao']) . ')';
                                    $ativoLabel = ($p['ativo'] == 1) ? ' [Ativo]' : '';
                                ?>
                                    <option value="<?= $p['id'] ?>" data-status="<?= htmlspecialchars($p['situacao']) ?>">
                                        <?= htmlspecialchars($p['personagem']) . $statusLabel . $ativoLabel ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div>
                            <label for="novo_jogador">Jogador de Destino:</label>
                            <input type="text" name="novo_jogador" id="novo_jogador" maxlength="25" required 
                                   placeholder="Nome exato do destinatário...">
                        </div>
                        
                        <div style="margin-top: 15px;">
                            <label for="pin">Código PIN:</label>
                            <input type="password" id="pin" name="pin" maxlength="4" required 
                                   placeholder="****" 
                                   style="width: 100%; padding: 15px; background-color: rgba(0, 0, 20, 0.8); border: 2px solid var(--primary); border-radius: 8px; color: white; font-size: 1.1rem; box-shadow: 0 0 5px rgba(0, 0, 255, 0.3); outline: none; transition: all 0.3s ease; text-align: center;">
                        </div>
                        
                        <div style="margin-top: 20px;">
                            <button type="submit" name="transferir_personagem" class="btn-transferir-neon">
                                TRANSFERIR AGORA
                            </button>
                        </div>
                    </form>
                </div>
            <?php endif; ?>
            </div> 
        <footer class="footer-simple-medieval">
            <div class="footer-content">
                <h4>⚔️ RPG ERA MEDIEVAL: FERRO & FOGO ⚔️</h4>
                <p>&copy; <?php echo date("Y"); ?> Todos os direitos reservados.</p>
            </div>
        </footer>

    </div> 
</body>
</html>