<?php
session_start();
require '../conexao.php';

// Verifica se o usuário está logado
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit();
}

$user_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name'];

// Configurações de paginação
$por_pagina = 10;
$pagina = isset($_GET['pagina']) ? max(1, intval($_GET['pagina'])) : 1;
$offset = ($pagina - 1) * $por_pagina;

// Processar transformação de personagem em ficha
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['transformar_ficha'])) {
    $personagem_id = $_POST['personagem_id'];
    $datahora_atual = date('Y-m-d H:i:s');
    
    try {
        $pdo->beginTransaction();
        
        // 1. Pegar TODOS os dados do personagem
        $stmt = $pdo->prepare("SELECT * FROM personagens WHERE id = ? AND jogador = ?");
        $stmt->execute([$personagem_id, $user_name]);
        $personagem = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$personagem) {
            throw new Exception("Personagem não encontrado!");
        }
        
        // 2. Gerar hash e link únicos
        $tentativas = 0;
        do {
            $hash = bin2hex(random_bytes(16));
            $link = bin2hex(random_bytes(8));
            
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM fichaspersonagens WHERE hash = ? OR link = ?");
            $stmt->execute([$hash, $link]);
            $tentativas++;
            
            if ($tentativas > 5) {
                throw new Exception("Não foi possível gerar identificadores únicos para a ficha");
            }
        } while ($stmt->fetchColumn() > 0);
        
        // 3. Criar a ficha primeiro para obter o ID
        $stmt = $pdo->prepare("INSERT INTO fichaspersonagens 
                             (jogador, situacao, hash, link, responsavel, data, descricao, preco, validade) 
                             VALUES (?, '1', ?, ?, ?, ?, ?, 0, ?)");
        $validade = date('Y-m-d H:i:s', strtotime('+1 year'));
        $stmt->execute([
            $user_name,
            $hash,
            $link,
            $user_name,
            $datahora_atual,
            "Ficha gerada do personagem {$personagem['personagem']}",
            $validade
        ]);
        
        $ficha_id = $pdo->lastInsertId();
        
        // 4. Pegar TODOS os dados da ficha recém-criada
        $stmt = $pdo->prepare("SELECT * FROM fichaspersonagens WHERE id = ?");
        $stmt->execute([$ficha_id]);
        $ficha = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // 5. Registrar no infossistema com TODOS os dados
        $informacao = "TRANSFORMAÇÃO DE PERSONAGEM EM FICHA\n\n";
        $informacao .= "DADOS DO JOGADOR:\n";
        $informacao .= "ID: $user_id\n";
        $informacao .= "Nome: $user_name\n\n";
        
        $informacao .= "DADOS DO PERSONAGEM:\n";
        foreach ($personagem as $campo => $valor) {
            $informacao .= "$campo: " . (is_null($valor) ? 'NULL' : $valor) . "\n";
        }
        
        $informacao .= "\nDADOS DA FICHA GERADA:\n";
        foreach ($ficha as $campo => $valor) {
            $informacao .= "$campo: " . (is_null($valor) ? 'NULL' : $valor) . "\n";
        }
        
        $informacao .= "\nData/Hora: $datahora_atual";
        
        $stmt = $pdo->prepare("INSERT INTO infossistema 
                             (informacao, p1, p2, origem, dataehora) 
                             VALUES (?, ?, ?, 'Transformação', ?)");
        $stmt->execute([
            $informacao,
            $user_name,
            $personagem['personagem'],
            $datahora_atual
        ]);
        
        // 6. Apagar o personagem
        $stmt = $pdo->prepare("DELETE FROM personagens WHERE id = ?");
        $stmt->execute([$personagem_id]);
        
        $pdo->commit();
        
        echo "<script>
                alert('Personagem transformado em ficha com sucesso!\\nID da Ficha: $ficha_id\\nHash: $hash');
                window.location.href = window.location.pathname;
              </script>";
        exit();
        
    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        echo "<script>alert('Erro: ".addslashes($e->getMessage())."');</script>";
    }
}

// Contar total de personagens
$stmt = $pdo->prepare("SELECT COUNT(*) FROM personagens WHERE jogador = ?");
$stmt->execute([$user_name]);
$total_personagens = $stmt->fetchColumn();

// Listar personagens do jogador com paginação (solução compatível com todas versões)
$sql = "SELECT id, personagem, soldados, ouro, prata, cargo, datacriacao 
        FROM personagens 
        WHERE jogador = ? 
        ORDER BY datacriacao DESC 
        LIMIT " . intval($por_pagina) . " OFFSET " . intval($offset);

$stmt = $pdo->prepare($sql);
$stmt->execute([$user_name]);
$personagens = $stmt->fetchAll(PDO::FETCH_ASSOC);

$total_paginas = ceil($total_personagens / $por_pagina);
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Transformar Personagem em Ficha</title>
    <link rel="stylesheet" type="text/css" href="../style.css">

     <style>
        /* Cores azuis vibrantes */
        :root {
            --azul-vivo: #0000FF;       /* Azul puro e vibrante */
            --azul-hover: #1E90FF;      /* Azul Dodger para hover */
            --azul-claro: #87CEFA;      /* Azul claro para detalhes */
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: #1a1a1a;
            color: #e0e0e0;
            margin: 0;
            padding: 20px;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background-color: #2a2a2a;
            border-radius: 8px;
            padding: 20px;
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.3);
        }
        
        h1, h2 {
            color: var(--azul-vivo);     /* Azul vivo puro */
            text-align: center;
            text-shadow: 0 0 5px rgba(0, 0, 255, 0.3); /* Efeito de brilho */
        }
        
        .personagem-card {
            background-color: #333;
            border-left: 4px solid var(--azul-vivo);
            border-radius: 4px;
            padding: 15px;
            margin-bottom: 10px;
            transition: all 0.3s;
        }
        
        .personagem-card:hover {
            transform: translateY(-3px);
            box-shadow: 0 4px 12px rgba(0, 0, 255, 0.2);
            background-color: #3a3a3a;
        }
        
        .personagem-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 10px;
        }
        
        .personagem-nome {
            font-size: 1.3em;
            font-weight: bold;
            color: var(--azul-vivo);
        }
        
        .personagem-detalhes {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
            gap: 10px;
            margin-top: 10px;
        }
        
        .detalhe-item {
            background-color: #444;
            padding: 5px 10px;
            border-radius: 4px;
            font-size: 0.9em;
        }
        
        .btn-transformar {
            background-color: var(--azul-vivo);
            color: white;
            border: none;
            padding: 8px 15px;
            border-radius: 4px;
            cursor: pointer;
            font-weight: bold;
            transition: all 0.3s;
        }
        
        .btn-transformar:hover {
            background-color: var(--azul-hover);
            transform: scale(1.05);
        }
        
        .paginacao {
            display: flex;
            justify-content: center;
            margin-top: 20px;
            gap: 5px;
        }
        
        .pagina-link {
            padding: 5px 10px;
            background-color: #333;
            color: #e0e0e0;
            text-decoration: none;
            border-radius: 4px;
            transition: all 0.3s;
        }
        
        .pagina-link:hover, .pagina-ativa {
            background-color: var(--azul-vivo);
            color: white;
        }
        
        .sem-personagens {
            text-align: center;
            padding: 20px;
            background-color: #333;
            border-radius: 4px;
        }

        /* Estilo para o botão do menu principal */
       

        /* Adicionando efeitos de brilho para mais vivacidade */
        .personagem-nome, h1, h2 {
            animation: pulse 3s infinite;
        }
        
        @keyframes pulse {
            0% { text-shadow: 0 0 5px rgba(0, 0, 255, 0.3); }
            50% { text-shadow: 0 0 15px rgba(0, 0, 255, 0.5); }
            100% { text-shadow: 0 0 5px rgba(0, 0, 255, 0.3); }
        }
    </style>

</head>
<body>
<center>
<img class="bn" src="../bn.png" alt=""><br><br>

<a href="../dashboard.php"><button class="btnmenu">Menu principal</button></a><br><br>

<a href="../admpersonagens.php"><button class="btnmenu">Administração de personagens</button></a><br><br><br>
    <div class="container">
        <h1>Transformar Personagem em Ficha</h1>
        
        <?php if (empty($personagens)): ?>
            <div class="sem-personagens">
                <p>Você não possui personagens para transformar em fichas.</p>
            </div>
        <?php else: ?>
            <h2>Seus Personagens (<?= $total_personagens ?> no total)</h2>
            
            <?php foreach ($personagens as $p): ?>
                <form method="POST" class="personagem-form">
                    <input type="hidden" name="personagem_id" value="<?= $p['id'] ?>">
                    <div class="personagem-card">
                        <div class="personagem-header">  
                            <div class="personagem-nome">Personagem: <?= htmlspecialchars($p['personagem']) ?></div>
                            <button type="submit" name="transformar_ficha" class="btn-transformar" 
                                    onclick="return confirm('Tem certeza que deseja transformar \'<?= addslashes($p['personagem']) ?>\' em uma ficha?\\nEsta ação é irreversível!')">
                                Transformar
                            </button>
                        </div>
                        <div class="personagem-detalhes">
                            <span class="detalhe-item">🛡️ Soldados: <?= $p['soldados'] ?></span>
                            <span class="detalhe-item">💰 Ouro: <?= $p['ouro'] ?></span>
                            <span class="detalhe-item">💍 Prata: <?= $p['prata'] ?></span>
                            <span class="detalhe-item">🎖️ Cargo: <?= $p['cargo'] ? htmlspecialchars($p['cargo']) : 'Nenhum' ?></span>
                            <span class="detalhe-item">📅 Criado em: <?= date('d/m/Y H:i', strtotime($p['datacriacao'])) ?></span>
                        </div>
                    </div>
                </form>
            <?php endforeach; ?>
            
            <?php if ($total_paginas > 1): ?>
                <div class="paginacao">
                    <?php if ($pagina > 1): ?>
                        <a href="?pagina=<?= $pagina-1 ?>" class="pagina-link">« Anterior</a>
                    <?php endif; ?>
                    
                    <?php 
                    // Mostrar até 5 páginas ao redor da atual
                    $inicio = max(1, $pagina - 2);
                    $fim = min($total_paginas, $pagina + 2);
                    
                    if ($inicio > 1) echo '<span class="pagina-link">...</span>';
                    for ($i = $inicio; $i <= $fim; $i++): ?>
                        <a href="?pagina=<?= $i ?>" class="pagina-link <?= $i == $pagina ? 'pagina-ativa' : '' ?>">
                            <?= $i ?>
                        </a>
                    <?php endfor;
                    if ($fim < $total_paginas) echo '<span class="pagina-link">...</span>';
                    ?>
                    
                    <?php if ($pagina < $total_paginas): ?>
                        <a href="?pagina=<?= $pagina+1 ?>" class="pagina-link">Próxima »</a>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</body>
</html>