<?php
session_start();
require '../conexao.php';

// 1. Verificação de Login
if (!isset($_SESSION['user_id'])) {
    header("Location: ../login.php");
    exit();
}

$comprador_user = $_SESSION['user_name'];
$comprador_wallet = $_SESSION['user_wallet'];
$id_ordem_venda = filter_input(INPUT_POST, 'id', FILTER_VALIDATE_INT);
$pin_digitado = $_POST['pin'] ?? ''; // Recebe o PIN

// Variáveis para exibir resultado
$sucesso = false;
$mensagem = "";

// 2. PROCESSAMENTO DA COMPRA
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['confirmar']) && $id_ordem_venda) {
    
    try {
        // --- PASSO DE SEGURANÇA: VERIFICAR PIN ---
        if (empty($pin_digitado)) {
            throw new Exception("O PIN é obrigatório para realizar a compra.");
        }
        
        // Verifica se o PIN bate com o da sessão (hash)
        if (!isset($_SESSION['user_pin']) || !password_verify($pin_digitado, $_SESSION['user_pin'])) {
            throw new Exception("PIN Incorreto! Transação negada.");
        }
        // ------------------------------------------

        // INÍCIO DA TRANSAÇÃO
        $pdo->beginTransaction();

        // A. Buscar a Ordem de Venda (Bloqueando para evitar concorrência)
        $stmtOrdem = $pdo->prepare("SELECT * FROM ordem_vendas WHERE id = :id LIMIT 1 FOR UPDATE");
        $stmtOrdem->execute([':id' => $id_ordem_venda]);
        $ordem = $stmtOrdem->fetch(PDO::FETCH_ASSOC);

        if (!$ordem) {
            throw new Exception("Esta oferta não existe mais (já foi vendida ou cancelada).");
        }

        $valor = $ordem['valor'];
        $personagem_nome = $ordem['personagem'];
        $vendedor_user = $ordem['jogador'];

        // B. Verificar Saldo do Comprador (Calculado na hora)
        $stmtSaldoIn = $pdo->prepare("SELECT SUM(valor) FROM mempool WHERE destinatario = :wallet");
        $stmtSaldoIn->execute([':wallet' => $comprador_wallet]);
        $entradas = $stmtSaldoIn->fetchColumn() ?: 0;

        $stmtSaldoOut = $pdo->prepare("SELECT SUM(valor) FROM mempool WHERE remetente = :wallet");
        $stmtSaldoOut->execute([':wallet' => $comprador_wallet]);
        $saidas = $stmtSaldoOut->fetchColumn() ?: 0;

        $saldo_real = bcsub($entradas, $saidas, 2);

        if ($saldo_real < $valor) {
            throw new Exception("Saldo insuficiente. Você tem R$ $saldo_real e precisa de R$ $valor.");
        }

        // C. Buscar Wallet do Vendedor
        $stmtVend = $pdo->prepare("SELECT wallet FROM usersblock WHERE user = :user");
        $stmtVend->execute([':user' => $vendedor_user]);
        $vendedor_wallet = $stmtVend->fetchColumn();

        if (!$vendedor_wallet) {
            throw new Exception("Erro ao identificar carteira do vendedor.");
        }

        // D. EXECUÇÃO
        
        // 1. Transferir Personagem
        $stmtPers = $pdo->prepare("UPDATE personagens SET jogador = :novo_dono, situacao = 'livre', datamovimentacao = NOW() WHERE personagem = :nome");
        $stmtPers->execute([':novo_dono' => $comprador_user, ':nome' => $personagem_nome]);

        // 2. Registrar Pagamento (Mempool)
        $data_hora = date('Y-m-d H:i:s');
        $referencia = "Compra de Personagem: $personagem_nome (Ordem #$id_ordem_venda)";
        $hash_base = $comprador_wallet . $vendedor_wallet . $valor . $data_hora . microtime();
        $hash = hash('sha256', $hash_base);
        $link = substr(hash('sha1', $hash), 0, 40);

        $sqlMempool = "INSERT INTO mempool (remetente, destinatario, valor, dataehora, referencia, hash, link) 
                       VALUES (:rem, :dest, :val, :data, :ref, :hash, :link)";
        $stmtMempool = $pdo->prepare($sqlMempool);
        $stmtMempool->execute([
            ':rem' => $comprador_wallet,
            ':dest' => $vendedor_wallet,
            ':val' => $valor,
            ':data' => $data_hora,
            ':ref' => $referencia,
            ':hash' => $hash,
            ':link' => $link
        ]);

        // 3. Registrar Estatística (Gráfico)
        $sqlVenda = "INSERT INTO vendas (quantidade, preco, dataehora, vendedor, comprador, referencia, mediacao) 
                     VALUES (1, :preco, :data, :vend, :comp, :ref, 'Mercado')";
        $stmtVenda = $pdo->prepare($sqlVenda);
        $stmtVenda->execute([
            ':preco' => $valor,
            ':data' => $data_hora,
            ':vend' => $vendedor_user,
            ':comp' => $comprador_user,
            ':ref' => $referencia
        ]);

        // 4. Log
        $logInfo = "O jogador $comprador_user comprou $personagem_nome de $vendedor_user por R$ $valor.";
        $sqlLog = "INSERT INTO infossistema (informacao, p1, p2, origem, dataehora) VALUES (:info, :p1, :p2, 'Compra Mercado', :data)";
        $stmtLog = $pdo->prepare($sqlLog);
        $stmtLog->execute([':info' => $logInfo, ':p1' => $comprador_user, ':p2' => $vendedor_user, ':data' => $data_hora]);

        // 5. Notificar Vendedor
        $notifMsg = "Seu personagem $personagem_nome foi vendido por R$ $valor para $comprador_user.";
        $sqlNotif = "INSERT INTO notificacoes_sis (tipo, notificacao, user1, user2, dataehora) VALUES ('venda', :msg, 'Sistema', :dest, :data)";
        $stmtNotif = $pdo->prepare($sqlNotif);
        $stmtNotif->execute([':msg' => $notifMsg, ':dest' => $vendedor_user, ':data' => $data_hora]);

        // 6. Apagar Ordem
        $stmtDel = $pdo->prepare("DELETE FROM ordem_vendas WHERE id = :id");
        $stmtDel->execute([':id' => $id_ordem_venda]);

        $pdo->commit();
        
        $sucesso = true;
        $mensagem = "Compra realizada com sucesso! O personagem <strong>$personagem_nome</strong> agora é seu.";

    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        $sucesso = false;
        $mensagem = "Erro: " . $e->getMessage();
    }
} else {
    header("Location: ordens_venda.php");
    exit();
}
?>

<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Resultado da Compra - Mavilla</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="../css/dashboard.css">
    <style>
        /* === ESTILO NEON TRANSPARENTE === */
        html, body { background: none !important; background-color: transparent !important; }
        body {
            background: url('../rpg-login-background.jpg') no-repeat center center fixed !important;
            background-size: cover !important;
            color: white;
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            font-family: 'Roboto', sans-serif;
        }

        .result-card {
            background-color: rgba(0, 5, 30, 0.95);
            border: 2px solid <?php echo $sucesso ? '#28a745' : '#ff2222'; ?>;
            box-shadow: 0 0 30px <?php echo $sucesso ? 'rgba(40, 167, 69, 0.4)' : 'rgba(255, 34, 34, 0.4)'; ?>;
            border-radius: 15px;
            padding: 40px;
            text-align: center;
            max-width: 600px;
            width: 90%;
            backdrop-filter: blur(5px);
        }

        .icon-result { font-size: 4rem; margin-bottom: 20px; color: <?php echo $sucesso ? '#28a745' : '#ff2222'; ?>; }
        
        h1 {
            color: <?php echo $sucesso ? '#28a745' : '#ff2222'; ?>;
            text-transform: uppercase;
            text-shadow: 0 0 10px <?php echo $sucesso ? '#28a745' : '#ff2222'; ?>;
            margin-bottom: 20px;
            font-size: 1.8rem;
        }

        p { font-size: 1.2rem; line-height: 1.6; color: #ddd; margin-bottom: 30px; }
        strong { color: var(--neon-bright); }

        .btn-back {
            background: #000; border: 2px solid var(--primary); color: #fff;
            padding: 12px 30px; border-radius: 8px; font-weight: bold;
            text-transform: uppercase; cursor: pointer; transition: 0.3s;
            text-decoration: none; display: inline-block;
        }
        .btn-back:hover { background: var(--primary); color: #000; box-shadow: 0 0 20px var(--neon-bright); }
    </style>
</head>
<body>

    <div class="result-card">
        <div class="icon-result">
            <?= $sucesso ? '✔' : '✖' ?>
        </div>
        
        <h1><?= $sucesso ? 'Compra Confirmada!' : 'Falha na Compra' ?></h1>
        
        <p><?= $mensagem ?></p>

        <div style="display: flex; gap: 10px; justify-content: center; flex-wrap: wrap;">
            <a href="ordens_venda.php" class="btn-back">Voltar ao Mercado</a>
            <?php if ($sucesso): ?>
                <a href="../dashboard.php" class="btn-back" style="border-color:#28a745; color:#28a745;">Ver Meu Personagem</a>
            <?php endif; ?>
        </div>
    </div>

</body>
</html>