<?php
session_start();
require '../conexao.php';

// 1. Verificação de Login
if (!isset($_SESSION['user_id'])) {
    header("Location: ../login.php");
    exit();
}

// 2. PROTEÇÃO CSRF (Adicionado)
if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
    $_SESSION['mensagem'] = "Erro de segurança: Sessão expirada ou inválida. Tente novamente.";
    header("Location: ordens_venda.php");
    exit();
}

$comprador_user = $_SESSION['user_name'];
$comprador_wallet = $_SESSION['user_wallet'];
$id_ordem_venda = filter_input(INPUT_POST, 'id', FILTER_VALIDATE_INT);
$pin_digitado = $_POST['pin'] ?? ''; 

$sucesso = false;
$mensagem = "";

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['confirmar']) && $id_ordem_venda) {
    
    try {
        // --- PASSO DE SEGURANÇA: VERIFICAR PIN ---
        if (empty($pin_digitado)) {
            throw new Exception("O PIN é obrigatório.");
        }
        
        if (!isset($_SESSION['user_pin']) || !password_verify($pin_digitado, $_SESSION['user_pin'])) {
             // Log de tentativa falha (Opcional, mas recomendado)
             // $stmtLog = $pdo->prepare("INSERT INTO logs_seguranca ...");
             throw new Exception("PIN Incorreto! Transação negada.");
        }

        // INÍCIO DA TRANSAÇÃO BLINDADA
        $pdo->beginTransaction();

        // A. Buscar a Ordem de Venda (Travando a linha)
        $stmtOrdem = $pdo->prepare("SELECT * FROM ordem_vendas WHERE id = :id LIMIT 1 FOR UPDATE");
        $stmtOrdem->execute([':id' => $id_ordem_venda]);
        $ordem = $stmtOrdem->fetch(PDO::FETCH_ASSOC);

        if (!$ordem) {
            throw new Exception("Esta oferta não existe mais.");
        }

        // CORREÇÃO: Força string para BCMath
        $valor = number_format((float)$ordem['valor'], 2, '.', '');
        $personagem_nome = $ordem['personagem'];
        $vendedor_user = $ordem['jogador'];

        // CORREÇÃO: Impedir Auto-Compra (Wash Trading)
        if ($vendedor_user === $comprador_user) {
            throw new Exception("Você não pode comprar seu próprio personagem.");
        }

        // B. Verificar Saldo do Comprador
        // NOTA: Se você tiver a tabela 'carteiras_saldos' (do código anterior), USE ELA aqui também.
        // Vou manter a lógica do mempool que você mandou, mas usando bcsub corretamente.
        
        $stmtSaldoIn = $pdo->prepare("SELECT SUM(valor) FROM mempool WHERE destinatario = :wallet");
        $stmtSaldoIn->execute([':wallet' => $comprador_wallet]);
        $entradas = $stmtSaldoIn->fetchColumn() ?: '0.00';

        $stmtSaldoOut = $pdo->prepare("SELECT SUM(valor) FROM mempool WHERE remetente = :wallet");
        $stmtSaldoOut->execute([':wallet' => $comprador_wallet]);
        $saidas = $stmtSaldoOut->fetchColumn() ?: '0.00';

        $saldo_real = bcsub($entradas, $saidas, 2);

        // CORREÇÃO: Comparação segura com bccomp
        // Se saldo (-1) for menor que valor
        if (bccomp($saldo_real, $valor, 2) === -1) {
             throw new Exception("Saldo insuficiente. Você tem R$ " . number_format($saldo_real, 2, ',', '.') . ".");
        }

        // C. Buscar Wallet do Vendedor
        $stmtVend = $pdo->prepare("SELECT wallet FROM usersblock WHERE user = :user");
        $stmtVend->execute([':user' => $vendedor_user]);
        $vendedor_wallet = $stmtVend->fetchColumn();

        if (!$vendedor_wallet) {
            throw new Exception("Erro técnica: Carteira do vendedor não encontrada.");
        }

        // D. EXECUÇÃO
        
        // 1. Transferir Personagem
        // Tenta pegar o ID do personagem se possível, senão vai pelo nome
        // IMPORTANTE: Adicionei LIMIT 1 para evitar atualizar múltiplos se houver nomes duplicados
        $stmtPers = $pdo->prepare("UPDATE personagens SET jogador = :novo_dono, situacao = 'livre', datamovimentacao = NOW() WHERE personagem = :nome AND jogador = :antigo_dono LIMIT 1");
        $stmtPers->execute([
            ':novo_dono' => $comprador_user, 
            ':nome' => $personagem_nome,
            ':antigo_dono' => $vendedor_user // Trava extra de segurança
        ]);
        
        if ($stmtPers->rowCount() == 0) {
            throw new Exception("Erro: O personagem não pertence mais ao vendedor especificado.");
        }

        // 2. Registrar Pagamento (Mempool)
        $data_hora = date('Y-m-d H:i:s');
        $referencia = "Compra: $personagem_nome (Ref #$id_ordem_venda)";
        $hash_base = $comprador_wallet . $vendedor_wallet . $valor . microtime();
        $hash = hash('sha256', $hash_base);
        $link = substr(hash('sha1', $hash), 0, 40);

        $sqlMempool = "INSERT INTO mempool (remetente, destinatario, valor, dataehora, referencia, hash, link) 
                       VALUES (:rem, :dest, :val, :data, :ref, :hash, :link)";
        $stmtMempool = $pdo->prepare($sqlMempool);
        $stmtMempool->execute([
            ':rem' => $comprador_wallet,
            ':dest' => $vendedor_wallet,
            ':val' => $valor, // String formatada
            ':data' => $data_hora,
            ':ref' => $referencia,
            ':hash' => $hash,
            ':link' => $link
        ]);
        
        // SE TIVER A TABELA carteiras_saldos, ATUALIZE ELA AQUI TAMBÉM (Recomendado)
        // $pdo->exec("UPDATE carteiras_saldos SET saldo = saldo - $valor WHERE wallet = '$comprador_wallet'");
        // $pdo->exec("UPDATE carteiras_saldos SET saldo = saldo + $valor WHERE wallet = '$vendedor_wallet'");

        // 3. Registrar Estatística
        $sqlVenda = "INSERT INTO vendas (quantidade, preco, dataehora, vendedor, comprador, referencia, mediacao) 
                     VALUES (1, :preco, :data, :vend, :comp, :ref, 'Mercado')";
        $stmtVenda = $pdo->prepare($sqlVenda);
        $stmtVenda->execute([
            ':preco' => $valor,
            ':data' => $data_hora,
            ':vend' => $vendedor_user,
            ':comp' => $comprador_user,
            ':ref' => $referencia
        ]);

        // 4. Logs e Notificações
        $logInfo = "$comprador_user comprou $personagem_nome de $vendedor_user por R$ $valor";
        $pdo->prepare("INSERT INTO infossistema (informacao, p1, p2, origem, dataehora) VALUES (?, ?, ?, 'Compra Mercado', NOW())")->execute([$logInfo, $comprador_user, $vendedor_user]);

        $notifMsg = "Seu personagem $personagem_nome foi vendido por R$ " . number_format($valor, 2, ',', '.') . " para $comprador_user.";
        $pdo->prepare("INSERT INTO notificacoes_sis (tipo, notificacao, user1, user2, dataehora) VALUES ('venda', ?, 'Sistema', ?, NOW())")->execute([$notifMsg, $vendedor_user]);

        // 5. Apagar Ordem
        $stmtDel = $pdo->prepare("DELETE FROM ordem_vendas WHERE id = :id");
        $stmtDel->execute([':id' => $id_ordem_venda]);

        $pdo->commit();
        
        // Renova o token CSRF após transação crítica para evitar reenvio
        unset($_SESSION['csrf_token']);
        
        $sucesso = true;
        $mensagem = "Compra realizada com sucesso!";

    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        $sucesso = false;
        $mensagem = "Erro: " . $e->getMessage();
    }
} else {
    header("Location: ordens_venda.php");
    exit();
}
?>