<?php
// confirmarcomprapers.php

require '../conexao.php';
require '../config_seguranca.php'; // ADICIONADO

// 1. Verificação de Login
if (!isset($_SESSION['user_id'])) {
    header("Location: ../login.php");
    exit();
}

// 2. BLINDAGEM: Verifica Token CSRF
verificarTokenCSRF(); 

$comprador_user = $_SESSION['user_name'];
$comprador_wallet = $_SESSION['user_wallet'];
$comprador_id = $_SESSION['user_id']; // Necessário para buscar o PIN no DB

$id_ordem_venda = filter_input(INPUT_POST, 'id', FILTER_VALIDATE_INT);
$pin_digitado = $_POST['pin'] ?? ''; 

$sucesso = false;
$mensagem = "";

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['confirmar']) && $id_ordem_venda) {
    
    try {
        // --- PASSO DE SEGURANÇA: VERIFICAR PIN ---
        if (empty($pin_digitado)) {
            throw new Exception("O PIN é obrigatório para realizar a compra.");
        }
        
        // SEGURANÇA MÁXIMA: Buscar hash do PIN direto do banco de dados
        // Não confiar na sessão para validação crítica de dinheiro
        $stmtPin = $pdo->prepare("SELECT pin FROM usersblock WHERE id = :uid LIMIT 1");
        $stmtPin->execute([':uid' => $comprador_id]);
        $hash_real_pin = $stmtPin->fetchColumn();

        if (!$hash_real_pin || !password_verify($pin_digitado, $hash_real_pin)) {
             // Delay para evitar Brute Force
             sleep(1);
             throw new Exception("PIN Incorreto! Transação negada.");
        }
        // ------------------------------------------

        // INÍCIO DA TRANSAÇÃO
        $pdo->beginTransaction();

        // A. Buscar a Ordem de Venda (TRAVANDO LINHA - FOR UPDATE)
        $stmtOrdem = $pdo->prepare("SELECT * FROM ordem_vendas WHERE id = :id LIMIT 1 FOR UPDATE");
        $stmtOrdem->execute([':id' => $id_ordem_venda]);
        $ordem = $stmtOrdem->fetch(PDO::FETCH_ASSOC);

        if (!$ordem) {
            throw new Exception("Esta oferta não existe mais (já foi vendida ou cancelada).");
        }

        $valor = $ordem['valor'];
        $personagem_nome = $ordem['personagem'];
        $vendedor_user = $ordem['jogador'];

        // VERIFICAÇÃO EXTRA: Wash Trading (Comprar de si mesmo)
        if ($vendedor_user === $comprador_user) {
             throw new Exception("Você não pode comprar seu próprio personagem.");
        }

        // B. Verificar Saldo do Comprador (Calculado na hora)
        // Mantive a lógica do mempool conforme solicitado
        $stmtSaldoIn = $pdo->prepare("SELECT SUM(valor) FROM mempool WHERE destinatario = :wallet");
        $stmtSaldoIn->execute([':wallet' => $comprador_wallet]);
        $entradas = $stmtSaldoIn->fetchColumn() ?: 0;

        $stmtSaldoOut = $pdo->prepare("SELECT SUM(valor) FROM mempool WHERE remetente = :wallet");
        $stmtSaldoOut->execute([':wallet' => $comprador_wallet]);
        $saidas = $stmtSaldoOut->fetchColumn() ?: 0;

        $saldo_real = bcsub($entradas, $saidas, 2);

        // Uso de bccomp para comparação precisa de float/dinheiro
        if (bccomp($saldo_real, $valor, 2) === -1) {
            throw new Exception("Saldo insuficiente. Você tem R$ " . number_format($saldo_real, 2, ',', '.') . ".");
        }

        // C. Buscar Wallet do Vendedor
        $stmtVend = $pdo->prepare("SELECT wallet FROM usersblock WHERE user = :user LIMIT 1");
        $stmtVend->execute([':user' => $vendedor_user]);
        $vendedor_wallet = $stmtVend->fetchColumn();

        if (!$vendedor_wallet) {
            throw new Exception("Erro ao identificar carteira do vendedor.");
        }

        // D. EXECUÇÃO
        
        // 1. Transferir Personagem (Garante que ainda é do vendedor na WHERE)
        $stmtPers = $pdo->prepare("UPDATE personagens SET jogador = :novo_dono, situacao = 'livre', datamovimentacao = NOW() WHERE personagem = :nome AND jogador = :antigo_dono");
        $stmtPers->execute([
            ':novo_dono' => $comprador_user, 
            ':nome' => $personagem_nome,
            ':antigo_dono' => $vendedor_user
        ]);
        
        if ($stmtPers->rowCount() == 0) {
            throw new Exception("Erro de integridade: O personagem mudou de dono durante a transação.");
        }

        // 2. Registrar Pagamento (Mempool)
        $data_hora = date('Y-m-d H:i:s');
        $referencia = "Compra de Personagem: $personagem_nome (Ordem #$id_ordem_venda)";
        // Melhorando entropia do hash
        $hash_base = $comprador_wallet . $vendedor_wallet . $valor . $data_hora . microtime(true) . bin2hex(random_bytes(10));
        $hash = hash('sha256', $hash_base);
        $link = substr(hash('sha1', $hash), 0, 40);

        $sqlMempool = "INSERT INTO mempool (remetente, destinatario, valor, dataehora, referencia, hash, link) 
                        VALUES (:rem, :dest, :val, :data, :ref, :hash, :link)";
        $stmtMempool = $pdo->prepare($sqlMempool);
        $stmtMempool->execute([
            ':rem' => $comprador_wallet,
            ':dest' => $vendedor_wallet,
            ':val' => $valor,
            ':data' => $data_hora,
            ':ref' => $referencia,
            ':hash' => $hash,
            ':link' => $link
        ]);
        
        // Se tiver tabela carteiras_saldos, atualize-a aqui também para performance!
        // UPDATE carteiras_saldos SET saldo = saldo - :val WHERE wallet = :rem
        // UPDATE carteiras_saldos SET saldo = saldo + :val WHERE wallet = :dest

        // 3. Registrar Estatística
        $sqlVenda = "INSERT INTO vendas (quantidade, preco, dataehora, vendedor, comprador, referencia, mediacao) 
                      VALUES (1, :preco, :data, :vend, :comp, :ref, 'Mercado')";
        $stmtVenda = $pdo->prepare($sqlVenda);
        $stmtVenda->execute([
            ':preco' => $valor,
            ':data' => $data_hora,
            ':vend' => $vendedor_user,
            ':comp' => $comprador_user,
            ':ref' => $referencia
        ]);

        // 4. Log Seguro
        $logInfo = "O jogador $comprador_user comprou $personagem_nome de $vendedor_user por R$ $valor.";
        $sqlLog = "INSERT INTO infossistema (informacao, p1, p2, origem, dataehora) VALUES (:info, :p1, :p2, 'Compra Mercado', :data)";
        $stmtLog = $pdo->prepare($sqlLog);
        $stmtLog->execute([':info' => $logInfo, ':p1' => $comprador_user, ':p2' => $vendedor_user, ':data' => $data_hora]);

        // 5. Notificar Vendedor
        $notifMsg = "Seu personagem $personagem_nome foi vendido por R$ $valor para $comprador_user.";
        $sqlNotif = "INSERT INTO notificacoes_sis (tipo, notificacao, user1, user2, dataehora) VALUES ('venda', :msg, 'Sistema', :dest, :data)";
        $stmtNotif = $pdo->prepare($sqlNotif);
        $stmtNotif->execute([':msg' => $notifMsg, ':dest' => $vendedor_user, ':data' => $data_hora]);

        // 6. Apagar Ordem
        $stmtDel = $pdo->prepare("DELETE FROM ordem_vendas WHERE id = :id");
        $stmtDel->execute([':id' => $id_ordem_venda]);

        $pdo->commit();
        
        // Limpa token usado para evitar duplo clique (Replay Attack)
        unset($_SESSION['csrf_token']);
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32)); // Gera novo imediatamente

        $sucesso = true;
        $mensagem = "Compra realizada com sucesso! O personagem <strong>$personagem_nome</strong> agora é seu.";

    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        $sucesso = false;
        // Não mostrar erro técnico detalhado em produção, apenas mensagem genérica ou tratada
        $mensagem = "Erro na transação: " . $e->getMessage();
    }
} else {
    header("Location: ordens_venda.php");
    exit();
}
?>
<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <title>Resultado da Compra - Mavilla</title>
    </head>
<body>
    <div class="result-card">
        <div class="icon-result">
            <?= $sucesso ? '✔' : '✖' ?>
        </div>
        <h1><?= $sucesso ? 'Compra Confirmada!' : 'Falha na Compra' ?></h1>
        <p><?= $mensagem ?></p>
        <div style="display: flex; gap: 10px; justify-content: center; flex-wrap: wrap;">
            <a href="ordens_venda.php" class="btn-back">Voltar ao Mercado</a>
            <?php if ($sucesso): ?>
                <a href="../dashboard.php" class="btn-back" style="border-color:#28a745; color:#28a745;">Ver Meu Personagem</a>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>