<?php
// transformar_personagem.php (SISTEMA BLINDADO & BLOCKCHAIN COMPATIBLE)

// 1. SEGURANÇA E CONEXÃO
require '../conexao.php';
require '../config_seguranca.php'; 

// 2. VERIFICAÇÃO DE LOGIN
if (!isset($_SESSION['user_id'])) {
    header("Location: ../login.php");
    exit();
}

$user_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name'];
$csrf_token = $_SESSION['csrf_token'];

// Captura mensagens de feedback
$mensagem_sucesso = "";
$mensagem_erro = "";

if (isset($_SESSION['msg_sucesso'])) {
    $mensagem_sucesso = $_SESSION['msg_sucesso'];
    unset($_SESSION['msg_sucesso']);
}
if (isset($_SESSION['msg_erro'])) {
    $mensagem_erro = $_SESSION['msg_erro'];
    unset($_SESSION['msg_erro']);
}

// Configurações de paginação
$por_pagina = 10;
$pagina = filter_input(INPUT_GET, 'pagina', FILTER_VALIDATE_INT);
$pagina = $pagina ? max(1, $pagina) : 1;
$offset = ($pagina - 1) * $por_pagina;

// --- PROCESSAMENTO DA TRANSFORMAÇÃO (POST BLINDADO) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['transformar_ficha'])) {
    
    // 3. BARREIRA CSRF
    verificarTokenCSRF();

    $personagem_id = filter_input(INPUT_POST, 'personagem_id', FILTER_VALIDATE_INT);
    $datahora_atual = date('Y-m-d H:i:s');
    
    if ($personagem_id) {
        try {
            $pdo->beginTransaction();
            
            // 4. SELEÇÃO COM TRAVA (FOR UPDATE) - ANTI-DUPLICAÇÃO
            $stmt = $pdo->prepare("SELECT * FROM personagens WHERE id = ? AND jogador = ? LIMIT 1 FOR UPDATE");
            $stmt->execute([$personagem_id, $user_name]);
            $personagem = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$personagem) {
                throw new Exception("Personagem não encontrado ou já transformado!");
            }
            
            // 5. Gerar hash e link únicos
            $tentativas = 0;
            do {
                $hash = bin2hex(random_bytes(16));
                $link = bin2hex(random_bytes(8));
                
                $stmtCheck = $pdo->prepare("SELECT COUNT(*) FROM fichaspersonagens WHERE hash = ? OR link = ?");
                $stmtCheck->execute([$hash, $link]);
                $tentativas++;
                
                if ($tentativas > 5) {
                    throw new Exception("Erro ao gerar identificadores únicos. Tente novamente.");
                }
            } while ($stmtCheck->fetchColumn() > 0);
            
            // 6. Criar a ficha
            $stmtInsert = $pdo->prepare("INSERT INTO fichaspersonagens 
                                         (jogador, situacao, hash, link, responsavel, data, descricao, preco, validade) 
                                         VALUES (?, '1', ?, ?, ?, ?, ?, 0, ?)");
            
            $validade = date('Y-m-d H:i:s', strtotime('+1 year'));
            // Descrição técnica interna
            $descricao_log = "Ficha gerada do personagem {$personagem['personagem']} (ID {$personagem['id']})";
            
            $stmtInsert->execute([
                $user_name,
                $hash,
                $link,
                $user_name,
                $datahora_atual,
                $descricao_log,
                $validade
            ]);
            
            $ficha_id = $pdo->lastInsertId();
            
            // --- 7. LOG NO INFOSSISTEMA (BLOCKCHAIN) ---
            
            // A. Buscar Wallet do Usuário (para anonimato no log público)
            $stmtW = $pdo->prepare("SELECT wallet FROM usersblock WHERE user = ? LIMIT 1");
            $stmtW->execute([$user_name]);
            $wallet_user = $stmtW->fetchColumn();
            
            if (!$wallet_user) $wallet_user = "WALLET_NOT_FOUND";

            // B. Montar Texto no Padrão Solicitado
            // "a wallet $wallet trocou o personagem $nomedopersonagem de id $idpersonagem pela ficha de id $idficha e $hashficha"
            $texto_blockchain = "Carteira $wallet_user trocou o personagem {$personagem['personagem']} de id {$personagem['id']} pela ficha de id $ficha_id e hash $hash";

            // C. Inserir no Infossistema
            // Origem definida como 'Transformação' para seu filtro SQL pegar depois
            $stmtLog = $pdo->prepare("INSERT INTO infossistema (informacao, p1, p2, origem, dataehora) VALUES (?, ?, ?, 'Transformação', ?)");
            // P1 = Wallet, P2 = Hash da Ficha (dados técnicos para busca rápida)
            $stmtLog->execute([$texto_blockchain, $wallet_user, $hash, $datahora_atual]);
            
            // 8. APAGAR O PERSONAGEM (Ação final irreversível)
            $stmtDel = $pdo->prepare("DELETE FROM personagens WHERE id = ?");
            $stmtDel->execute([$personagem_id]);
            
            $pdo->commit();
            
            // Mensagem de Sucesso Visual
            $_SESSION['msg_sucesso'] = "Personagem transformado com sucesso!<br>ID da Ficha: <strong>$ficha_id</strong>";
            header("Location: " . $_SERVER['PHP_SELF'] . "?pagina=" . $pagina);
            exit();
            
        } catch (Exception $e) {
            if ($pdo->inTransaction()) $pdo->rollBack();
            $_SESSION['msg_erro'] = "Erro: " . $e->getMessage();
            header("Location: " . $_SERVER['PHP_SELF'] . "?pagina=" . $pagina);
            exit();
        }
    }
}

// --- CONSULTAS DE LEITURA (VISUALIZAÇÃO) ---

// Contar total
$stmt = $pdo->prepare("SELECT COUNT(*) FROM personagens WHERE jogador = :jogador");
$stmt->execute([':jogador' => $user_name]);
$total_personagens = $stmt->fetchColumn();

// Listar personagens (Seguro com Binds)
$sql = "SELECT id, personagem, soldados, ouro, prata, cargo, datacriacao 
        FROM personagens 
        WHERE jogador = :jogador 
        ORDER BY datacriacao DESC 
        LIMIT :limit OFFSET :offset";

$stmt = $pdo->prepare($sql);
$stmt->bindValue(':jogador', $user_name);
$stmt->bindValue(':limit', $por_pagina, PDO::PARAM_INT);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->execute();
$personagens = $stmt->fetchAll(PDO::FETCH_ASSOC);

$total_paginas = ceil($total_personagens / $por_pagina);
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Transformar Personagem em Ficha - Ferro & Fogo</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="../css/dashboard.css">
</head>
<body>
    <div class="light-rays"></div>

    <div class="container"> 
        
        <div class="card" style="margin-top: 30px; margin-bottom: 50px;">
            
            <div class="listing-header-centered">
                <h1 style="color: var(--primary); font-family: 'Times New Roman', serif; text-transform: uppercase; letter-spacing: 2px; text-shadow: var(--text-neon-glow); margin-bottom: 25px; font-size: 2rem;">
                    RPG ERA MEDIEVAL: Ferro & Fogo Pelo Poder
                </h1>
                
                <div class="btn-group btn-container-center" style="margin-bottom: 25px;"><center>
                    <a href="../admpersonagens.php">
                        <button style="margin:0; padding: 12px 50px; width: auto;">Administração de Fichas & Personagens</button>
                    </a><br>
                    <a href="../dashboard.php">
                        <button style="margin:0; padding: 12px 50px; width: auto;">Menu Principal</button>
                    </a>
                    </center>
                </div>

                <h2 style="color: white; text-transform: uppercase; letter-spacing: 1px; margin-bottom: 15px;">
                    Transformar Personagem em Ficha
                </h2>

                <p class="listing-intro">
                    <b><font color="yellow">Atenção:</font> <font color="red">Esta ação é irreversível! Ao transformar um personagem em ficha, ele será excluído permanentemente e irreversivelmente, e você receberá uma ficha.</font></b>
                </p>
            </div>
            
            <?php if ($mensagem_sucesso): ?>
                <div class="message-neon success" style="border: 1px solid #00ff00; background: rgba(0,255,0,0.1); color: #00ff00; padding: 15px; margin: 10px auto; max-width: 600px; text-align: center; border-radius: 8px;">
                    <?= $mensagem_sucesso ?>
                </div>
            <?php endif; ?>
            
            <?php if ($mensagem_erro): ?>
                <div class="message-neon error" style="border: 1px solid #ff2222; background: rgba(255,0,0,0.1); color: #ff2222; padding: 15px; margin: 10px auto; max-width: 600px; text-align: center; border-radius: 8px;">
                    <?= htmlspecialchars($mensagem_erro) ?>
                </div>
            <?php endif; ?>

            <br>
            <div class="personagens-list-container">
                
                <?php if (empty($personagens)): ?>
                    <div class="sem-personagens-neon">
                        <p>Você não possui personagens para transformar em fichas.</p>
                    </div>
                <?php else: ?>
                    
                    <?php foreach ($personagens as $p): ?>
                        <form method="POST">
                            <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                            <input type="hidden" name="personagem_id" value="<?= $p['id'] ?>">
                            
                            <div class="personagem-neon-card" style="border-left-color: var(--primary);">
                                
                                <div class="personagem-header-neon">
                                    <div class="personagem-nome-neon" style="color: var(--primary);">
                                        Personagem: <?= htmlspecialchars($p['personagem']) ?>
                                    </div>
                                    
                                    <button type="submit" name="transformar_ficha" class="btn-transformar-neon" 
                                            onclick="return confirm('ATENÇÃO EXTREMA:\n\nTem certeza absoluta que deseja transformar o personagem \'<?= addslashes($p['personagem']) ?>\' em uma ficha?\n\nEsta ação é IRREVERSÍVEL! O personagem será excluído e você receberá uma ficha.')">
                                        Transformar
                                    </button>
                                </div>
                                
                                <div class="personagem-detalhes-grid">
                                    <div class="detalhe-item-neon"><span class="icon">🛡️</span> Soldados: <?= $p['soldados'] ?></div>
                                    <div class="detalhe-item-neon"><span class="icon">💰</span> Ouro: <?= $p['ouro'] ?></div>
                                    <div class="detalhe-item-neon"><span class="icon">💍</span> Prata: <?= $p['prata'] ?></div>
                                    <div class="detalhe-item-neon"><span class="icon">🎖️</span> Cargo: <?= $p['cargo'] ? htmlspecialchars($p['cargo']) : 'Nenhum' ?></div>
                                    <div class="detalhe-item-neon"><span class="icon">📅</span> Criado em: <?= date('d/m/Y H:i', strtotime($p['datacriacao'])) ?></div>
                                </div>

                            </div> 
                        </form>
                    <?php endforeach; ?>
                    
                    <?php if ($total_paginas > 1): ?>
                        <div class="pagination-neon">
                            <?php if ($pagina > 1): ?>
                                <a href="?pagina=<?= $pagina-1 ?>">« Anterior</a>
                            <?php endif; ?>
                            
                            <?php 
                            $inicio = max(1, $pagina - 2);
                            $fim = min($total_paginas, $pagina + 2);
                            
                            if ($inicio > 1) echo '<span style="color:var(--primary); padding: 10px;">...</span>';
                            
                            for ($i = $inicio; $i <= $fim; $i++): ?>
                                <?php if ($i == $pagina): ?>
                                    <a href="?pagina=<?= $i ?>" class="active"><?= $i ?></a>
                                <?php else: ?>
                                    <a href="?pagina=<?= $i ?>"><?= $i ?></a>
                                <?php endif; ?>
                            <?php endfor;
                            
                            if ($fim < $total_paginas) echo '<span style="color:var(--primary); padding: 10px;">...</span>';
                            ?>
                            
                            <?php if ($pagina < $total_paginas): ?>
                                <a href="?pagina=<?= $pagina+1 ?>">Próxima »</a>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>

                <?php endif; ?>
            </div> 
        </div> 
        <footer class="footer-simple-medieval">
            <div class="footer-content">
                <h4>⚔️ RPG ERA MEDIEVAL: FERRO & FOGO ⚔️</h4>
                <p>&copy; <?php echo date("Y"); ?> Todos os direitos reservados.</p>
            </div>
        </footer>

    </div> 
</body>
</html>