<?php
session_start();
// Em produção, mantenha 0.
ini_set('display_errors', 0); 
error_reporting(E_ALL);

require_once '../conexao.php';

// --- 1. VERIFICAÇÕES DE SEGURANÇA INICIAIS ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_wallet'])) {
    header("Location: ../login.php");
    exit();
}

// Verifica CSRF
if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
    $_SESSION['mensagem'] = "Erro de segurança: Sessão expirada. Tente novamente.";
    header('Location: mercado.php');
    exit;
}

// Apenas POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    die("Acesso inválido.");
}

// --- DADOS DO USUÁRIO ---
$remetente_wallet = $_SESSION['user_wallet'];
$remetente_nome   = $_SESSION['user_name'];
$ip_usuario       = $_SERVER['REMOTE_ADDR'];

// --- 2. TRATAMENTO INTELIGENTE DO VALOR (Versão Segura BCMath) ---
$destinatario = preg_replace('/[^a-zA-Z0-9]/', '', $_POST['destinatario'] ?? '');
$valor_post = $_POST['valor'] ?? '0';
$pin_digitado = $_POST['pin'] ?? '';

// Lógica de correção monetária (Mantida):
if (strpos($valor_post, ',') !== false) {
    $valor_post = str_replace('.', '', $valor_post);
    $valor_post = str_replace(',', '.', $valor_post);
}

// AQUI ESTÁ A CORREÇÃO:
// Formata para string "0.00" para o BCMath funcionar. Não usamos float puro.
$valor = number_format((float)$valor_post, 2, '.', '');

// --- 3. VALIDAÇÕES BÁSICAS ---
// bccomp: retorna 0 se igual, 1 se maior, -1 se menor.
// Se valor for menor ou igual a 0.00
if (bccomp($valor, '0.00', 2) <= 0) {
    $_SESSION['mensagem'] = "Erro: Valor inválido ou zerado.";
    header('Location: mercado.php');
    exit;
}

if ($destinatario === $remetente_wallet) {
    $_SESSION['mensagem'] = "Erro: Não é permitido enviar para si mesmo.";
    header('Location: mercado.php');
    exit;
}

if (empty($pin_digitado)) {
    $_SESSION['mensagem'] = "Erro: O PIN é obrigatório.";
    header('Location: mercado.php');
    exit;
}

// --- 4. INÍCIO DA TRANSAÇÃO BLINDADA ---
try {
    $pdo->beginTransaction();

    // A. Verifica PIN
    if (!isset($_SESSION['user_pin'])) {
        throw new Exception("Sessão inválida. Faça login novamente.");
    }
   
    if (!password_verify($pin_digitado, $_SESSION['user_pin'])) {
        // Log de erro silencioso
        $stmt = $pdo->prepare("INSERT INTO tentativas_pin (ip, jogador, local, timestamp) VALUES (:ip, :jog, 'envio_dinheiro', NOW())");
        $stmt->execute([':ip' => $ip_usuario, ':jog' => $remetente_wallet]);
       
        $pdo->commit();
        $_SESSION['mensagem'] = "Erro: PIN incorreto.";
        header('Location: mercado.php');
        exit;
    }

    // B. Verifica e Trava Saldo (Tabela Segura)
    $stmt = $pdo->prepare("SELECT saldo FROM carteiras_saldos WHERE wallet = :wallet FOR UPDATE");
    $stmt->execute([':wallet' => $remetente_wallet]);
    $saldo_atual = $stmt->fetchColumn();

    // Auto-migração (Mantida igual, pois já estava correta usando bcsub)
    if ($saldo_atual === false) {
        $stmtOld = $pdo->prepare("SELECT SUM(valor) FROM mempool WHERE destinatario = :w");
        $stmtOld->execute([':w' => $remetente_wallet]);
        $entrou = $stmtOld->fetchColumn() ?: 0;
       
        $stmtOld2 = $pdo->prepare("SELECT SUM(valor) FROM mempool WHERE remetente = :w");
        $stmtOld2->execute([':w' => $remetente_wallet]);
        $saiu = $stmtOld2->fetchColumn() ?: 0;
       
        $saldo_calculado = bcsub($entrou, $saiu, 2);
        $saldo_calculado = max(0, $saldo_calculado);
       
        $stmtIns = $pdo->prepare("INSERT INTO carteiras_saldos (wallet, saldo) VALUES (:w, :s)");
        $stmtIns->execute([':w' => $remetente_wallet, ':s' => $saldo_calculado]);
       
        $saldo_atual = $saldo_calculado;
       
        // Re-trava
        $stmt = $pdo->prepare("SELECT saldo FROM carteiras_saldos WHERE wallet = :wallet FOR UPDATE");
        $stmt->execute([':wallet' => $remetente_wallet]);
    }

    // AQUI É A SEGURANÇA MÁXIMA:
    // Verifica se tem dinheiro usando BCMath (String vs String)
    // Se saldo_atual for MENOR (-1) que o valor
    if (bccomp($saldo_atual, $valor, 2) === -1) {
        throw new Exception("Saldo insuficiente.");
    }

    // O código segue normalmente para "C. Verifica Destinatário"...

    // C. Verifica Destinatário
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM usersblock WHERE wallet = :dest");
    $stmt->execute([':dest' => $destinatario]);
    if ($stmt->fetchColumn() == 0) {
        throw new Exception("Destinatário não encontrado.");
    }

    // Trava carteira Destino
    $stmt = $pdo->prepare("SELECT saldo FROM carteiras_saldos WHERE wallet = :dest FOR UPDATE");
    $stmt->execute([':dest' => $destinatario]);
    $saldo_dest = $stmt->fetchColumn();

    if ($saldo_dest === false) {
        $stmt = $pdo->prepare("INSERT INTO carteiras_saldos (wallet, saldo) VALUES (:dest, 0.00)");
        $stmt->execute([':dest' => $destinatario]);
    }

    // D. Executa a Transferência
    $stmt = $pdo->prepare("UPDATE carteiras_saldos SET saldo = saldo - :valor WHERE wallet = :rem");
    $stmt->execute([':valor' => $valor, ':rem' => $remetente_wallet]);

    $stmt = $pdo->prepare("UPDATE carteiras_saldos SET saldo = saldo + :valor WHERE wallet = :dest");
    $stmt->execute([':valor' => $valor, ':dest' => $destinatario]);

    // E. Gera Histórico (Mempool e Infossistema)
    $hash = hash('sha256', $destinatario . $remetente_wallet . $valor . microtime());
    $link = bin2hex(random_bytes(25));
    
    // --- NOVIDADE: FORMATAR VALOR PARA 2 CASAS DECIMAIS ---
    $valor_formatado = number_format($valor, 2, ',', '.'); 
    
    // Texto de referência corrigido, usando $valor_formatado
    $referencia = "A carteira $remetente_wallet enviou R$ $valor_formatado para a carteira $destinatario";

    $stmt = $pdo->prepare("INSERT INTO mempool (destinatario, remetente, dataehora, valor, hash, link, referencia) VALUES (:d, :r, NOW(), :v, :h, :l, :ref)");
    // OBSERVAÇÃO: A variável :v (valor) ainda usa $valor (float)
    // porque o campo no banco (DECIMAL) precisa do float puro (ex: 3.00)
    $stmt->execute([':d'=>$destinatario, ':r'=>$remetente_wallet, ':v'=>$valor, ':h'=>$hash, ':l'=>$link, ':ref'=>$referencia]);

    // Infossistema
    $stmt = $pdo->prepare("INSERT INTO infossistema (informacao, p1, p2, origem, dataehora) VALUES (:i, :p1, :p2, 'Envio de dinheiro', NOW())");
    $stmt->execute([':i'=>$referencia, ':p1'=>$remetente_nome, ':p2'=>$destinatario]);

    // F. Finaliza
    $pdo->commit();

    unset($_SESSION['csrf_token']);
    
    // --- MENSAGEM RESTAURADA CONFORME ORIGINAL (Já estava correta) ---
    // Esta linha já usava number_format e está OK.
    $_SESSION['mensagem'] = "Transação de R$ " . $valor_formatado . " enviada com sucesso<br>Para a carteira:<br> $destinatario";
    $_SESSION['linkhash'] = $link;
    
    header('Location: mercado.php');
    exit;

} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    $_SESSION['mensagem'] = "Erro: " . $e->getMessage();
    header('Location: mercado.php');
    exit;
}
?>