<?php
session_start();
require '../../conex/conexao.php';

// 1. Segurança Básica
if (!isset($_SESSION['personagem']) || !isset($_POST['direcao'])) {
    header("Location: dashboardmed.php");
    exit();
}

$personagem_id = $_SESSION['personagem'];
$id2_destino = filter_input(INPUT_POST, 'direcao', FILTER_VALIDATE_INT);
$agora = date('Y-m-d H:i:s');

try {
    // INÍCIO DA TRANSAÇÃO (Essencial para escalabilidade e ACID)
    $pdo->beginTransaction();

    // 2. Busca dados do Personagem com TRAVAMENTO (FOR UPDATE)
    // Isso impede que duas requisições processem o movimento ao mesmo tempo (Race Condition)
    $sqlChar = "SELECT local, paralisado, movido FROM personagens WHERE id = :id FOR UPDATE";
    $stmtChar = $pdo->prepare($sqlChar);
    $stmtChar->execute([':id' => $personagem_id]);
    $char = $stmtChar->fetch(PDO::FETCH_ASSOC);

    if (!$char) {
        throw new Exception("Personagem não encontrado.");
    }

    // 3. Verificações de Bloqueio (Paralisia e Cooldown)
    // Verifica Paralisia
    if (!empty($char['paralisado']) && $char['paralisado'] > $agora) {
        $tempo = date('H:i:s', strtotime($char['paralisado']));
        $_SESSION['msg'] = "Você está paralisado até às $tempo!";
        $pdo->rollBack();
        header("Location: ../../dashboardmed.php");
        exit();
    }

    // Verifica Cooldown (15 segundos)
    if (!empty($char['movido'])) {
        $segundosPassados = time() - strtotime($char['movido']);
        if ($segundosPassados < 15) {
            $restante = 15 - $segundosPassados;
            $_SESSION['msg'] = "Aguarde $restante segundos para andar novamente.";
            $pdo->rollBack();
            header("Location: ../../dashboardmed.php");
            exit();
        }
    }

    // 4. Descobre a Região e o Destino
    // Primeiro, precisamos saber a região do local ATUAL
    $sqlMapaAtual = "SELECT regiao FROM mapasmed WHERE nome = :nome LIMIT 1";
    $stmtAtual = $pdo->prepare($sqlMapaAtual);
    $stmtAtual->execute([':nome' => $char['local']]);
    $mapaAtual = $stmtAtual->fetch(PDO::FETCH_ASSOC);

    if (!$mapaAtual) {
        throw new Exception("Local atual bugado/inexistente.");
    }

    $regiao = $mapaAtual['regiao'];

    // Agora buscamos o alvo usando o ID2 (direção) e a Região
    $sqlAlvo = "SELECT nome, tipo FROM mapasmed WHERE id2 = :id2 AND regiao = :regiao LIMIT 1";
    $stmtAlvo = $pdo->prepare($sqlAlvo);
    $stmtAlvo->execute([':id2' => $id2_destino, ':regiao' => $regiao]);
    $alvo = $stmtAlvo->fetch(PDO::FETCH_ASSOC);

    // 5. Verificações de Destino
    if (!$alvo) {
        $_SESSION['msg'] = "Caminho bloqueado ou inexistente.";
        $pdo->rollBack();
        header("Location: ../../dashboardmed.php");
        exit();
    }

    // --- NOVA REGRA: BLOQUEIO DE MAR ---
    if ($alvo['tipo'] === 'mar') {
        $_SESSION['msg'] = "Você não pode entrar no mar sem um barco!";
        $pdo->rollBack();
        header("Location: ../../dashboardmed.php");
        exit();
    }
    // -----------------------------------

    // 6. Executa o Movimento
    $novo_local = $alvo['nome'];
    $fome = $_SESSION['fome'];
    $sede = $_SESSION['sede'];
   $fome = max(0, $fome - 2); 
$sede = max(0, $sede - 3);

    // Atualiza Personagem
    $sqlUpdate = "UPDATE personagens SET local = :novo_local, movido = :agora, fome = :fome, sede = :sede WHERE id = :id";
    $stmtUpdate = $pdo->prepare($sqlUpdate);
    $stmtUpdate->execute([
        ':novo_local' => $novo_local,
        ':agora'      => $agora,
        ':fome'         => $fome,
        ':sede'         => $sede,
         ':id'         => $personagem_id
    ]);
    
    // Opcional: Inserir no Log de Movimentos (tabela movpersonagens)
    // Isso ajuda a rastrear hacks ou bugs futuramente
    /*
    $sqlLog = "INSERT INTO movpersonagens (personagem, origem, destino, jogador, data) VALUES (:pid, :origem, :destino, :jog, NOW())";
    $stmtLog = $pdo->prepare($sqlLog);
    $stmtLog->execute([
        ':pid' => $personagem_id,
        ':origem' => $char['local'],
        ':destino' => $novo_local,
        ':jog' => $_SESSION['user_name'] ?? 'Sistema'
    ]);
    */

    // Confirma a Transação
    $pdo->commit();

    // 7. Atualiza Sessão e Redireciona Instantaneamente
    $_SESSION['local'] = $novo_local;
    $_SESSION['msg'] = "Você viajou para $novo_local."; // Opcional, pode remover se quiser menos spam visual

    header("Location: ../../dashboardmed.php");
    exit();

} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    $_SESSION['msg'] = "Erro ao mover: " . $e->getMessage();
    header("Location: ../../dashboardmed.php");
    exit();
}
?>