<?php
session_start();
require '../conexao.php';

// 1. SEGURANÇA BÁSICA
if (!isset($_SESSION['user_id'])) {
    header("Location: ../login.php");
    exit();
}

$user_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name'];
$user_wallet = $_SESSION['user_wallet'];

// Se não for POST, volta
if (!isset($_POST['criarOrdem'])) {
    header("Location: mercado.php");
    exit();
}

// 2. VALIDAÇÃO CSRF E PIN
if ($_POST['csrf_token'] !== $_SESSION['csrf_token']) {
    $_SESSION['mensagem'] = "Erro de Segurança: Token Inválido.";
    header('Location: mercado.php');
    exit;
}

// Verificação de Bloqueio de PIN
$stmt = $pdo->prepare("SELECT COUNT(*) FROM tentativas_pin WHERE jogador = ? AND timestamp >= NOW() - INTERVAL 24 HOUR");
$stmt->execute([$user_name]);
if ($stmt->fetchColumn() >= 7) {
    $_SESSION['mensagem'] = "PIN BLOQUEADO por segurança. Aguarde 24h.";
    header('Location: mercado.php');
    exit;
}

$pin_digitado = $_POST['pin'];
if (!password_verify($pin_digitado, $_SESSION['user_pin'])) {
    $stmt = $pdo->prepare("INSERT INTO tentativas_pin (ip, jogador, local, timestamp) VALUES (?, ?, 'criar_ordem', NOW())");
    $stmt->execute([$_SERVER['REMOTE_ADDR'], $user_name]);
    $_SESSION['mensagem'] = 'PIN Incorreto.';
    header('Location: mercado.php');
    exit;
}

// 3. RECEBE E VALIDA DADOS
$quantidade_total = (int)$_POST['quantidade'];
$valor_unitario = filter_var(str_replace(',', '.', $_POST['valor']), FILTER_VALIDATE_FLOAT);

if ($quantidade_total <= 0 || $valor_unitario <= 0) {
    $_SESSION['mensagem'] = 'Quantidade ou Valor inválidos.';
    header('Location: mercado.php');
    exit;
}

$custo_total = bcmul($valor_unitario, $quantidade_total, 2);

// 4. VERIFICAR SALDO
$stmtIn = $pdo->prepare("SELECT SUM(valor) FROM mempool WHERE destinatario = ?");
$stmtIn->execute([$user_wallet]);
$entrada = $stmtIn->fetchColumn() ?: 0;

$stmtOut = $pdo->prepare("SELECT SUM(valor) FROM mempool WHERE remetente = ?");
$stmtOut->execute([$user_wallet]);
$saida = $stmtOut->fetchColumn() ?: 0;

$saldo_real = bcsub($entrada, $saida, 2);

if ($saldo_real < $custo_total) {
    $_SESSION['mensagem'] = "Saldo insuficiente. Você tem R$ " . number_format($saldo_real, 2, ',', '.') . " e precisa de R$ " . number_format($custo_total, 2, ',', '.');
    header("Location: mercado.php");
    exit;
}

// ==================================================================================
// 5. BUSCA DE OFERTAS (MATCHING)
// ==================================================================================
$sqlPersonagens = "SELECT id, jogador, personagem AS nome, valor, 1 AS qtd, 'personagem' AS tipo 
                   FROM ordem_vendas 
                   WHERE valor <= :valor AND jogador != :eu";

$sqlFichas = "SELECT id, jogador, 'Ficha' AS nome, preco AS valor, 1 AS qtd, 'ficha' AS tipo 
              FROM fichaspersonagens 
              WHERE preco <= :valor AND situacao = 'vendendo' AND jogador != :eu";

$stmtP = $pdo->prepare($sqlPersonagens);
$stmtP->execute([':valor' => $valor_unitario, ':eu' => $user_name]);
$ofertas_personagens = $stmtP->fetchAll(PDO::FETCH_ASSOC);

$stmtF = $pdo->prepare($sqlFichas);
$stmtF->execute([':valor' => $valor_unitario, ':eu' => $user_name]);
$ofertas_fichas = $stmtF->fetchAll(PDO::FETCH_ASSOC);

$ofertas = array_merge($ofertas_personagens, $ofertas_fichas);

usort($ofertas, function ($a, $b) {
    if ($a['valor'] == $b['valor']) return 0;
    return ($a['valor'] < $b['valor']) ? -1 : 1;
});

// ==================================================================================
// 6. PROCESSAMENTO DA COMPRA (LOOP)
// ==================================================================================

$quantidade_faltante = $quantidade_total;
$total_gasto = 0;
$detalhes_compras = []; 
$data_hora = date('Y-m-d H:i:s');

try {
    $pdo->beginTransaction();

    // Vamos criar um ID de Ordem "Virtual" ou "Real" para referência
    // Como a ordem é processada instantaneamente, podemos usar o timestamp como ID lógico para o log
    $id_ordem_logica = "AUTO-" . time(); 

    foreach ($ofertas as $oferta) {
        if ($quantidade_faltante <= 0) break; 

        $vendedor = $oferta['jogador'];
        $preco_item = $oferta['valor'];
        $tipo_item = $oferta['tipo'];
        $id_oferta_original = $oferta['id']; 
        $nome_item = $oferta['nome'];

        // Transferência do Item
        if ($tipo_item == 'personagem') {
            $stmtCheck = $pdo->prepare("SELECT id FROM ordem_vendas WHERE id = ? FOR UPDATE");
            $stmtCheck->execute([$id_oferta_original]);
            if (!$stmtCheck->fetch()) continue; 

            $updPers = $pdo->prepare("UPDATE personagens SET jogador = ?, situacao = 'livre', datamovimentacao = NOW() WHERE personagem = ?");
            $updPers->execute([$user_name, $nome_item]);
            
            $delVenda = $pdo->prepare("DELETE FROM ordem_vendas WHERE id = ?");
            $delVenda->execute([$id_oferta_original]);

        } else {
            $stmtCheck = $pdo->prepare("SELECT id FROM fichaspersonagens WHERE id = ? AND situacao = 'vendendo' FOR UPDATE");
            $stmtCheck->execute([$id_oferta_original]);
            if (!$stmtCheck->fetch()) continue;

            $updFicha = $pdo->prepare("UPDATE fichaspersonagens SET jogador = ?, situacao = '1', preco = 0 WHERE id = ?");
            $updFicha->execute([$user_name, $id_oferta_original]);
        }

        // Pagamento (Mempool)
        $stmtW = $pdo->prepare("SELECT wallet FROM usersblock WHERE user = ?");
        $stmtW->execute([$vendedor]);
        $wallet_vendedor = $stmtW->fetchColumn();

        $ref = "Venda Automática ($tipo_item): $nome_item (Ref Ordem: $id_ordem_logica)";
        $hash = hash('sha256', $user_wallet . $wallet_vendedor . $preco_item . microtime());
        $link = substr(hash('sha1', $hash), 0, 40);

        $insMem = $pdo->prepare("INSERT INTO mempool (remetente, destinatario, valor, dataehora, referencia, hash, link) VALUES (?, ?, ?, ?, ?, ?, ?)");
        $insMem->execute([$user_wallet, $wallet_vendedor, $preco_item, $data_hora, $ref, $hash, $link]);
        
        $id_transacao_mempool = $pdo->lastInsertId(); 

        // Registro de Venda
        $insVenda = $pdo->prepare("INSERT INTO vendas (quantidade, preco, dataehora, vendedor, comprador, referencia, mediacao) VALUES (1, ?, ?, ?, ?, ?, 'Ordem Compra')");
        $insVenda->execute([$preco_item, $data_hora, $vendedor, $user_name, $ref]);

        // --- CORREÇÃO: NOTIFICAÇÃO DETALHADA AO VENDEDOR ---
        $msgNotif = "Seu $tipo_item ($nome_item), que estava anunciado, foi vendido para uma Ordem de Compra (Ref: $id_ordem_logica). Valor da venda: R$ " . number_format($preco_item, 2, ',', '.') . ". Você recebeu o valor através da transação #$id_transacao_mempool.";
        
        $insNotif = $pdo->prepare("INSERT INTO notificacoes_sis (tipo, notificacao, user1, user2, dataehora) VALUES ('venda', ?, 'Sistema', ?, ?)");
        $insNotif->execute([$msgNotif, $vendedor, $data_hora]);
        // ---------------------------------------------------

        $quantidade_faltante--;
        $total_gasto += $preco_item;
        
        $detalhes_compras[] = [
            'tipo' => $tipo_item,
            'nome' => $nome_item,
            'vendedor' => $vendedor,
            'valor' => $preco_item,
            'id_ordem' => $id_oferta_original,
            'id_transacao' => $id_transacao_mempool
        ];
    }

    // 7. CRIAR ORDEM PARA O RESTANTE
    $id_nova_ordem = null;
    if ($quantidade_faltante > 0) {
        $valor_restante = bcmul($valor_unitario, $quantidade_faltante, 2);
        $ref_ordem = "Reserva para Ordem de Compra: $quantidade_faltante itens a R$ $valor_unitario";
        $hash_ordem = hash('sha256', $user_wallet . 'SISTEMA_ORDEM' . $valor_restante . microtime());
        
        $insMemOrdem = $pdo->prepare("INSERT INTO mempool (remetente, destinatario, valor, dataehora, referencia, hash) VALUES (?, 'SISTEMA_RESERVA', ?, ?, ?, ?)");
        $insMemOrdem->execute([$user_wallet, $valor_restante, $data_hora, $ref_ordem, $hash_ordem]);

        $insOrdem = $pdo->prepare("INSERT INTO ordens_compra (quantidade, valor, wallet, user, dataehora) VALUES (?, ?, ?, ?, ?)");
        $insOrdem->execute([$quantidade_faltante, $valor_unitario, $user_wallet, $user_name, $data_hora]);
        $id_nova_ordem = $pdo->lastInsertId();
    }

    $pdo->commit();

    // === RELATÓRIO FINAL ===
    $texto_copia = "RELATÓRIO DE OPERAÇÃO - MAVILLA\n";
    $texto_copia .= "Pedido: $quantidade_total itens até R$ " . number_format($valor_unitario, 2, ',', '.') . "\n";
    
    $qtd_comprada = $quantidade_total - $quantidade_faltante;
    
    $html_resumo = "<div style='text-align:left;'>";
    $html_resumo .= "<p>Você criou uma ordem de compra de <strong>$quantidade_total NFT's</strong>, por até <strong>R$ " . number_format($valor_unitario, 2, ',', '.') . "</strong>.</p>";
    
    if ($qtd_comprada > 0) {
        $html_resumo .= "<div class='status-box success'><p>✅ <strong>$qtd_comprada NFT's comprados imediatamente</strong> (Total Gasto: R$ " . number_format($total_gasto, 2, ',', '.') . ").</p></div>";
        
        $html_resumo .= "<div class='receipt-scroll'>"; 
        foreach ($detalhes_compras as $d) {
            $tipo_fmt = ucfirst($d['tipo']);
            $html_resumo .= "<div class='receipt-item'>";
            $html_resumo .= "<div class='receipt-header'>$tipo_fmt: <span style='color:#00f3ff'>{$d['nome']}</span></div>";
            $html_resumo .= "<div class='receipt-row'><span>Vendedor:</span> <span>{$d['vendedor']}</span></div>";
            $html_resumo .= "<div class='receipt-row'><span>Valor Pago:</span> <span style='color:#28a745; font-weight:bold;'>R$ " . number_format($d['valor'], 2, ',', '.') . "</span></div>";
            $html_resumo .= "<div class='receipt-row'><span>ID Ordem Venda:</span> <span class='code'>#{$d['id_ordem']}</span></div>";
            $html_resumo .= "<div class='receipt-row'><span>ID Transação:</span> <span class='code'>#{$d['id_transacao']}</span></div>";
            $html_resumo .= "</div>";

            $texto_copia .= "----------------\n";
            $texto_copia .= "Item: {$d['nome']} ({$d['tipo']})\n";
            $texto_copia .= "Valor: R$ " . number_format($d['valor'], 2, ',', '.') . "\n";
            $texto_copia .= "Vendedor: {$d['vendedor']}\n";
            $texto_copia .= "ID Transação: #{$d['id_transacao']}\n";
        }
        $html_resumo .= "</div>"; 
    }
    
    if ($quantidade_faltante > 0) {
        $html_resumo .= "<div class='status-box info'><p>📢 Uma <strong>Ordem de Compra (#$id_nova_ordem)</strong> foi criada para os <strong>$quantidade_faltante</strong> itens restantes.</p></div>";
        $texto_copia .= "\nPENDÊNCIA:\nOrdem #$id_nova_ordem criada para $quantidade_faltante itens restantes.\n";
    }
    
    $html_resumo .= "</div>";
    
    $texto_copia_js = json_encode($texto_copia);

    $_SESSION['resultado_html'] = $html_resumo;
    $_SESSION['texto_copia'] = $texto_copia_js;

} catch (Exception $e) {
    if ($pdo->inTransaction()) $pdo->rollBack();
    $_SESSION['mensagem'] = "Erro ao processar: " . $e->getMessage();
    header("Location: mercado.php");
    exit;
}
?>
<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Processamento - Mavilla</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="../css/dashboard.css">
    <style>
        body {
            background: url('../rpg-login-background.jpg') no-repeat center center fixed;
            background-size: cover;
            display: flex; align-items: center; justify-content: center; min-height: 100vh; color: white;
        }
        .neon-card {
            background-color: rgba(0, 5, 30, 0.95);
            border: 2px solid var(--primary);
            box-shadow: 0 0 30px rgba(0,0,255,0.4);
            border-radius: 15px;
            padding: 30px;
            width: 90%; max-width: 600px;
            text-align: center;
        }
        .btn-back {
            display: inline-block; padding: 10px 25px; background: #000; border: 2px solid var(--primary); color: #fff;
            border-radius: 6px; font-weight: bold; text-transform: uppercase; text-decoration: none; margin-top: 20px;
        }
        .btn-back:hover { background: var(--primary); color: #000; }
        .btn-copy { background: #28a745; border-color: #28a745; margin-bottom: 10px; }
        .btn-copy:hover { background: #218838; color: #fff; box-shadow: 0 0 15px #28a745; }
        
        /* Estilos para a lista de rolagem */
        .receipt-scroll { max-height: 300px; overflow-y: auto; background: rgba(0,0,0,0.5); border: 1px solid #333; border-radius: 8px; padding: 10px; margin-bottom: 20px; }
        .receipt-item { background: rgba(255,255,255,0.03); border-left: 3px solid var(--primary); padding: 10px; margin-bottom: 10px; border-radius: 0 4px 4px 0; }
        .receipt-header { font-weight: bold; font-size: 1rem; margin-bottom: 5px; text-transform: uppercase; }
        .receipt-row { display: flex; justify-content: space-between; font-size: 0.85rem; color: #ccc; margin-bottom: 2px; }
        .code { font-family: monospace; color: #888; }
        .status-box { padding: 10px; border-radius: 6px; margin: 15px 0; font-size: 0.95rem; }
        .status-box.success { background: rgba(40, 167, 69, 0.1); border: 1px solid #28a745; color: #28a745; }
        .status-box.info { background: rgba(0, 243, 255, 0.1); border: 1px solid #00f3ff; color: #00f3ff; }
    </style>
</head>
<body>
    <div class="neon-card">
        <?= $_SESSION['resultado_html'] ?>
        <div style="margin-top: auto;">
            <button onclick="copiarDados()" class="btn-back btn-copy"><i class="fas fa-copy"></i> Copiar Detalhes</button>
            <a href="mercado.php" class="btn-back">Voltar ao Mercado</a>
        </div>
    </div>
    <script>
        function copiarDados() {
            const texto = <?= $_SESSION['texto_copia'] ?>;
            navigator.clipboard.writeText(texto).then(function() { alert('Copiado!'); });
        }
    </script>
</body>
</html>