<?php
session_start();
require '../conexao.php';

// 1. Verificação de Login
if (!isset($_SESSION['user_id'])) {
    header("Location: ../login.php");
    exit();
}

$user_name = $_SESSION['user_name'];
$id_ordem = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
$erro = "";
$ordem = null;

// 2. BUSCAR DADOS COM SEGURANÇA (Verifica se o dono é quem está logado)
if ($id_ordem) {
    $sql = "SELECT * FROM ordem_vendas WHERE id = :id AND jogador = :dono";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([':id' => $id_ordem, ':dono' => $user_name]);
    $ordem = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$ordem) {
        $_SESSION['mensagem'] = "Erro: Ordem não encontrada ou você não tem permissão para cancelá-la.";
        header("Location: ordens_venda.php");
        exit();
    }
} else {
    header("Location: ordens_venda.php");
    exit();
}

// 3. PROCESSAR O CANCELAMENTO
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['confirmar'])) {
    try {
        $pdo->beginTransaction();

        // A. Remove a ordem de venda
        $stmtDel = $pdo->prepare("DELETE FROM ordem_vendas WHERE id = :id");
        $stmtDel->execute([':id' => $id_ordem]);

        // B. Libera o personagem (Volta a ser 'livre')
        $stmtUpd = $pdo->prepare("UPDATE personagens SET situacao = 'livre' WHERE personagem = :nome");
        $stmtUpd->execute([':nome' => $ordem['personagem']]);

        $pdo->commit();

        $_SESSION['mensagem'] = "Sucesso: A venda do personagem " . htmlspecialchars($ordem['personagem']) . " foi cancelada.";
        header("Location: ordens_venda.php");
        exit();

    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        $erro = "Erro ao cancelar: " . $e->getMessage();
    }
}
?>

<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cancelar Venda - Mavilla</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="../css/dashboard.css">
    
    <style>
        /* === ESTILO NEON TRANSPARENTE === */
        html, body { background: none !important; background-color: transparent !important; }
        
        body {
            background: url('../rpg-login-background.jpg') no-repeat center center fixed !important;
            background-size: cover !important;
            color: white;
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            font-family: 'Roboto', sans-serif;
        }

        .container { width: 100%; max-width: 500px; padding: 20px; }

        /* Card Central */
        .neon-card {
            background-color: rgba(0, 5, 30, 0.9); /* Vidro escuro */
            border: 2px solid #ff2222; /* Borda Vermelha (Alerta) */
            box-shadow: 0 0 20px rgba(255, 34, 34, 0.4);
            border-radius: 15px;
            padding: 30px;
            text-align: center;
            backdrop-filter: blur(5px);
        }

        h1 {
            color: #ff2222;
            text-transform: uppercase;
            text-shadow: 0 0 10px #ff2222;
            font-family: 'Times New Roman', serif;
            margin-bottom: 20px;
            border-bottom: 1px solid #ff2222;
            padding-bottom: 10px;
            font-size: 1.8rem;
        }

        .info-box {
            background: rgba(255,255,255,0.05);
            padding: 20px;
            border-radius: 8px;
            margin: 20px 0;
            text-align: left;
            border: 1px dashed #555;
        }
        
        .info-box p { margin: 8px 0; font-size: 1.1rem; color:#ccc; }
        .highlight { color: var(--neon-bright); font-weight: bold; font-size: 1.2rem; }
        .price { color: #28a745; font-weight: bold; font-size: 1.2rem; }

        /* Botões */
        .btn-group { display: flex; flex-direction: column; gap: 15px; margin-top: 30px; }
        
        button {
            width: 100%; padding: 15px; border-radius: 8px; font-weight: bold;
            text-transform: uppercase; cursor: pointer; transition: 0.3s; font-size: 1rem;
        }

        .btn-confirm {
            background: #000; color: #ff2222; border: 2px solid #ff2222;
        }
        .btn-confirm:hover {
            background: #ff2222; color: #fff; box-shadow: 0 0 20px #ff2222;
        }

        .btn-cancel {
            background: #000; color: var(--primary); border: 2px solid var(--primary);
        }
        .btn-cancel:hover {
            background: var(--primary); color: #fff; box-shadow: 0 0 20px var(--primary);
        }

    </style>
</head>
<body>

    <div class="container">
        <div class="neon-card">
            <h1>Cancelar Venda</h1>
            
            <?php if ($erro): ?>
                <div style="color:#ff2222; background:rgba(255,0,0,0.1); padding:10px; border-radius:5px; margin-bottom:15px;">
                    <?= $erro ?>
                </div>
            <?php endif; ?>

            <p>Deseja retirar este personagem do mercado?</p>

            <div class="info-box">
                <p>Personagem: <span class="highlight"><?= htmlspecialchars($ordem['personagem']) ?></span></p>
                <p>Valor Atual: <span class="price">R$ <?= number_format($ordem['valor'], 2, ',', '.') ?></span></p>
            </div>

            <form method="POST" class="btn-group">
                <input type="hidden" name="id" value="<?= $id_ordem ?>">
                
                <!-- Botão Confirmar -->
                <button type="submit" name="confirmar" class="btn-confirm">
                    Sim, Cancelar Venda
                </button>
                
                <!-- Botão Voltar -->
                <a href="ordens_venda.php" style="text-decoration:none; width:100%;">
                    <button type="button" class="btn-cancel">Não, Voltar</button>
                </a>
            </form>
        </div>
    </div>

</body>
</html>