<?php
session_start();
require '../conexao.php';

// Verifica login
if (!isset($_SESSION['user_id'])) {
    header("Location: ../login.php");
    exit();
}

$jogador_logado = $_SESSION['user_name'];

// --- CONFIGURAÇÃO PAGINAÇÃO ---
$itensPorPagina = 10;
$paginaAtual = isset($_GET['pagina']) ? max(1, intval($_GET['pagina'])) : 1;
$offset = ($paginaAtual - 1) * $itensPorPagina;

try {
    // 1. Contar total de grupos de preço (para paginação correta)
    $sqlCount = "SELECT COUNT(DISTINCT preco) FROM fichaspersonagens WHERE situacao = 'vendendo'";
    $totalItens = $pdo->query($sqlCount)->fetchColumn();
    $totalPaginas = ceil($totalItens / $itensPorPagina);

    // 2. Buscar dados agrupados com paginação no SQL (Muito mais rápido)
    // A lógica aqui é agrupar por preço e mostrar quantas fichas tem naquele preço
    // Também verifica se há fichas do jogador logado naquele preço
    $sql = "SELECT preco, COUNT(*) as quantidade, 
            SUM(CASE WHEN jogador = :eu THEN 1 ELSE 0 END) as minhas_qtd 
            FROM fichaspersonagens 
            WHERE situacao = 'vendendo'
            GROUP BY preco
            ORDER BY preco ASC
            LIMIT :limit OFFSET :offset";
            
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':eu', $jogador_logado, PDO::PARAM_STR);
    $stmt->bindValue(':limit', $itensPorPagina, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $fichas = $stmt->fetchAll(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    $erro = "Erro ao listar fichas: " . $e->getMessage();
}

function linkPagina($pag) { return "?pagina=" . $pag; }
?>

<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Fichas à Venda - Mavilla</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="../css/dashboard.css">
    
    <style>
        /* === ESTILOS ESPECÍFICOS === */
        html, body { background: none !important; background-color: transparent !important; }
        body {
            background: url('../rpg-login-background.jpg') no-repeat center center fixed !important;
            background-size: cover !important;
            color: white;
            min-height: 100vh;
        }

        .container { max-width: 1000px; margin: 0 auto; padding: 20px; }

        /* TABELA NEON */
        .table-container {
            overflow-x: auto;
            border: 2px solid var(--primary);
            border-radius: 10px;
            box-shadow: 0 0 20px rgba(0,0,255,0.2);
            background: rgba(0, 0, 0, 0.85);
            margin-top: 20px;
        }

        table { width: 100%; border-collapse: collapse; color: white; min-width: 600px; }
        thead { background-color: rgba(0, 0, 255, 0.2); border-bottom: 2px solid var(--primary); }
        th { padding: 15px; text-align: center; text-transform: uppercase; font-size: 0.9rem; color: var(--neon-bright); text-shadow: 0 0 5px var(--primary); }
        td { padding: 12px; text-align: center; border-bottom: 1px solid rgba(255,255,255,0.1); font-size: 1rem; }
        tr:last-child td { border-bottom: none; }
        tr:hover { background-color: rgba(255,255,255,0.05); }

        /* Tags */
        .price-tag { color: #28a745; font-weight: bold; font-size: 1.1rem; text-shadow: 0 0 5px rgba(40,167,69,0.5); }
        .qty-tag { font-weight: bold; color: #00f3ff; font-size: 1.1rem; }
        .mine-tag { color: #ffd700; font-size: 0.8rem; font-style: italic; }

        /* Destaque Minhas Fichas */
        .row-mine {
            background-color: rgba(0, 0, 255, 0.15);
            border-left: 4px solid var(--primary);
        }

        /* Botões */
        .btn-table {
            padding: 8px 15px; border-radius: 4px; font-weight: bold; text-transform: uppercase;
            font-size: 0.8rem; cursor: pointer; border: 1px solid transparent; transition: 0.3s; background: transparent;
        }
        .btn-buy { color: #28a745; border-color: #28a745; }
        .btn-buy:hover { background: #28a745; color: #000; box-shadow: 0 0 15px #28a745; }
        
        .btn-cancel { color: #ff2222; border-color: #ff2222; }
        .btn-cancel:hover { background: #ff2222; color: #fff; box-shadow: 0 0 15px #ff2222; }

        .btn-back {
            display: inline-block; padding: 10px 30px; background: #000; border: 2px solid var(--primary); color: #fff;
            border-radius: 8px; font-weight: bold; text-transform: uppercase; cursor: pointer; transition: 0.3s; text-decoration: none;
        }
        .btn-back:hover { background: var(--primary); color: #000; box-shadow: 0 0 15px var(--neon-bright); }

        /* Paginação */
        .pagination { display: flex; justify-content: center; gap: 5px; margin-top: 30px; flex-wrap: wrap; }
        .page-link {
            padding: 8px 12px; background: #000; border: 1px solid var(--primary); color: var(--primary);
            text-decoration: none; border-radius: 4px; font-weight: bold; font-size: 0.9rem;
        }
        .page-link:hover { background: var(--primary); color: #fff; box-shadow: 0 0 10px var(--primary); }
        .page-link.active { background: var(--primary); color: #fff; box-shadow: 0 0 15px var(--neon-bright); }
        .page-link.disabled { opacity: 0.5; pointer-events: none; border-color: #555; color: #555; }
        
        .msg-box {
            background: rgba(0,0,0,0.8); border: 1px solid #ccc; padding: 15px; border-radius: 8px; margin-bottom: 20px; text-align: center;
        }
    </style>
</head>
<body>

    <div class="container">
        
        <div class="listing-header-centered" style="text-align:center; margin-bottom:30px;">
            <h1 style="font-size: 2.5rem; color:var(--primary); text-shadow:0 0 15px var(--primary); margin-bottom: 20px;">
                FICHAS À VENDA
            </h1>
            <div style="display:flex; justify-content:center; gap:10px;">
                <a href="../mercado/mercado.php" class="btn-back">Voltar ao Mercado</a>
                <a href="vender_fichas.php" class="btn-back" style="border-color:#ffd700; color:#ffd700;">Vender Fichas</a>
            </div>
        </div>

        <?php if (isset($_SESSION['msg']) && !empty($_SESSION['msg'])): ?>
            <div class="msg-box">
                <?= htmlspecialchars($_SESSION['msg']) ?>
            </div>
            <?php unset($_SESSION['msg']); ?>
        <?php endif; ?>

        <?php if (isset($erro)): ?>
            <div class="msg-box" style="border-color:red; color:red;"><?= $erro ?></div>
        <?php endif; ?>

        <?php if (count($fichas) > 0): ?>
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>Preço Unitário</th>
                            <th>Quantidade</th>
                            <th>Ação</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($fichas as $row): 
                            // Se tenho fichas minhas nesse preço, destaca e mostra opção de cancelar
                            $tenho_minhas = ($row['minhas_qtd'] > 0);
                            $classe_linha = $tenho_minhas ? 'row-mine' : '';
                        ?>
                            <tr class="<?= $classe_linha ?>">
                                <td><span class="price-tag">R$ <?= number_format($row['preco'], 2, ',', '.') ?></span></td>
                                <td>
                                    <span class="qty-tag"><?= $row['quantidade'] ?></span>
                                    <?php if ($tenho_minhas): ?>
                                        <br><span class="mine-tag">(<?= $row['minhas_qtd'] ?> são suas)</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($row['quantidade'] > $row['minhas_qtd']): ?>
                                        <a href="confirmar_compra.php?preco=<?= $row['preco'] ?>">
                                            <button class="btn-table btn-buy">
                                                <i class="fas fa-shopping-cart"></i> Comprar
                                            </button>
                                        </a>
                                    <?php endif; ?>

                                    <?php if ($tenho_minhas): ?>
                                        <form method="POST" action="cancelar_venda.php" style="display:inline-block; margin-left:5px;">
                                            <input type="hidden" name="preco" value="<?= $row['preco'] ?>">
                                            <button type="submit" class="btn-table btn-cancel" onclick="return confirm('Deseja cancelar a venda de TODAS as suas <?= $row['minhas_qtd'] ?> fichas neste preço?');">
                                                <i class="fas fa-trash"></i> Cancelar Minhas
                                            </button>
                                        </form>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <?php if ($totalPaginas > 1): ?>
                <div class="pagination">
                    <?php if ($paginaAtual > 1): ?>
                        <a href="<?= linkPagina(1) ?>" class="page-link">« Primeira</a>
                        <a href="<?= linkPagina($paginaAtual - 1) ?>" class="page-link">‹ Anterior</a>
                    <?php else: ?>
                        <span class="page-link disabled">« Primeira</span>
                        <span class="page-link disabled">‹ Anterior</span>
                    <?php endif; ?>

                    <span class="page-link active">Página <?= $paginaAtual ?> de <?= $totalPaginas ?></span>

                    <?php if ($paginaAtual < $totalPaginas): ?>
                        <a href="<?= linkPagina($paginaAtual + 1) ?>" class="page-link">Próxima ›</a>
                        <a href="<?= linkPagina($totalPaginas) ?>" class="page-link">Última »</a>
                    <?php else: ?>
                        <span class="page-link disabled">Próxima ›</span>
                        <span class="page-link disabled">Última »</span>
                    <?php endif; ?>
                </div>
            <?php endif; ?>

        <?php else: ?>
            <div style="text-align:center; padding:50px; color:#aaa; font-style:italic; background:rgba(0,0,0,0.6); border:1px dashed #555; border-radius:10px;">
                <i class="fas fa-ticket-alt fa-3x"></i><br><br>
                Nenhuma ficha à venda no momento.
            </div>
        <?php endif; ?>

    </div>
</body>
</html>