<?php
session_start();
// Ajuste o caminho do require para voltar 2 pastas até a raiz
require '../../conexao.php'; 

// 1. Verificação de Acesso Básica
if (!isset($_SESSION['user_id']) || empty($_SESSION['personagem'])) {
    header("Location: ../../login.php");
    exit();
}

$user_name = $_SESSION['user_name'];
$nomePersonagem = $_SESSION['personagem_nome'];
$nome_cla = $_SESSION['cla'] ?? '';

// Redireciona se não tiver Clã na sessão (ou busca no banco pra garantir)
if (empty($nome_cla) || $nome_cla === 'Nenhum') {
    $_SESSION['msg'] = "Você precisa de um clã.";
    header("Location: ../../dashboard.php");
    exit();
}

// ==========================================================
// 2. BUSCA DE DADOS REAIS (Correção do Cargo)
// ==========================================================
// Vamos buscar o ID do Clã e o Cargo do personagem diretamente no banco
// para garantir que não pegue dados desatualizados da sessão.

// 2.1 Pegar ID do Clã
$stmtCla = $pdo->prepare("SELECT id FROM cla WHERE nome = :nome");
$stmtCla->execute([':nome' => $nome_cla]);
$cla_data = $stmtCla->fetch(PDO::FETCH_ASSOC);

if (!$cla_data) {
    echo "Erro: Clã não encontrado.";
    exit;
}
$cla_id_db = $cla_data['id'];

// 2.2 Pegar Cargo ATUAL do Personagem
$stmtPers = $pdo->prepare("SELECT cargo FROM personagens WHERE personagem = :nome");
$stmtPers->execute([':nome' => $nomePersonagem]);
$pers_data = $stmtPers->fetch(PDO::FETCH_ASSOC);

// Se por algum motivo o campo cargo for nulo, assume 1
$cargo_id = $pers_data['cargo'] ?? 1;

// Mapeamento de Cargos
$cargos_cla = [
    1 => 'Guardião da Vanguarda', 
    2 => 'Capitão do Mato', 
    3 => 'Capitão da Vanguarda',
    4 => 'Comandante da Guarda Real', 
    5 => 'Subcomandante', 
    6 => 'Comandante Oficial', 
    7 => '1º Ministro'
];
$cargo_nome = $cargos_cla[$cargo_id] ?? 'Recruta';

// Definição de Permissões
$pode_escrever_ordem = $cargo_id >= 3;
$pode_escolher_tipo = $cargo_id >= 6; // Só cargo 6+ escolhe entre Geral ou Maior
$pode_ver_historico = $cargo_id >= 6;

// ==========================================================
// 3. PROCESSAR ENVIO DE ORDEM
// ==========================================================
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $pode_escrever_ordem) {
    $ordem = trim($_POST['ordem'] ?? '');
    $tipo = $_POST['tipo'] ?? 'G';
    
    // Trava de segurança: Se cargo for menor que 6, força ser Geral
    if ($cargo_id < 6) $tipo = 'G';

    if (!empty($ordem)) {
        try {
            $sqlInsert = "INSERT INTO cla_ordens (cla_id, ordem, tipo, cargo, redator, data) 
                          VALUES (:cid, :ord, :tip, :carg, :red, NOW())";
            $stmtInsert = $pdo->prepare($sqlInsert);
            $stmtInsert->execute([
                ':cid' => $cla_id_db,
                ':ord' => $ordem,
                ':tip' => $tipo,
                ':carg' => $cargo_id,
                ':red' => $nomePersonagem
            ]);
            $_SESSION['msg'] = "Ordem publicada no pergaminho do clã!";
        } catch (Exception $e) {
            $_SESSION['msg'] = "Erro ao publicar ordem.";
        }
    }
    header("Location: ordens_cla.php");
    exit();
}

// ==========================================================
// 4. BUSCAR AS ÚLTIMAS ORDENS (Para exibição no Pergaminho)
// ==========================================================
// Buscar a última Ordem Geral
$stmtOG = $pdo->prepare("SELECT * FROM cla_ordens WHERE cla_id = :id AND tipo = 'G' ORDER BY data DESC LIMIT 1");
$stmtOG->execute([':id' => $cla_id_db]);
$ultima_og = $stmtOG->fetch(PDO::FETCH_ASSOC);

// Buscar a última Ordem Maior (M)
$stmtOM = $pdo->prepare("SELECT * FROM cla_ordens WHERE cla_id = :id AND tipo = 'M' ORDER BY data DESC LIMIT 1");
$stmtOM->execute([':id' => $cla_id_db]);
$ultima_om = $stmtOM->fetch(PDO::FETCH_ASSOC);

// ==========================================================
// 5. BUSCAR HISTÓRICO (Apenas para autorizados)
// ==========================================================
$historico = [];
if ($pode_ver_historico) {
    $limite = 10;
    $pagina = isset($_GET['p']) ? (int)$_GET['p'] : 1;
    $offset = ($pagina - 1) * $limite;
    
    $stmtHist = $pdo->prepare("SELECT * FROM cla_ordens WHERE cla_id = :id ORDER BY data DESC LIMIT :lim OFFSET :off");
    $stmtHist->bindParam(':id', $cla_id_db, PDO::PARAM_INT);
    $stmtHist->bindParam(':lim', $limite, PDO::PARAM_INT);
    $stmtHist->bindParam(':off', $offset, PDO::PARAM_INT);
    $stmtHist->execute();
    $historico = $stmtHist->fetchAll(PDO::FETCH_ASSOC);
}
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Pergaminhos do Clã | <?= htmlspecialchars($nome_cla) ?></title>
    <link href="https://fonts.googleapis.com/css2?family=Cinzel:wght@400;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="../../css/dashboardmed.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        .page-grid {
            display: grid;
            grid-template-columns: 1fr;
            gap: 25px;
            max-width: 900px;
            margin: 0 auto;
        }

        /* --- ESTILO PERGAMINHO NEON (NOVO) --- */
        .parchment-section {
            display: grid;
            grid-template-columns: 1fr 1fr; /* Dois pergaminhos lado a lado */
            gap: 20px;
        }
        
        .parchment-box {
            position: relative;
            background: #f0e6d2; /* Cor de papel envelhecido */
            background-image: url('https://www.transparenttextures.com/patterns/aged-paper.png'); /* Textura sutil opcional */
            color: #2c1b0e; /* Cor de tinta escura (Marrom café) */
            padding: 25px;
            border-radius: 5px;
            min-height: 250px;
            font-family: 'Cinzel', serif; /* Fonte Medieval */
            
            /* Borda Neon Azul envolvendo o papel */
            border: 3px solid var(--primary);
            box-shadow: 0 0 15px var(--primary), inset 0 0 20px rgba(0,0,0,0.1);
            
            display: flex;
            flex-direction: column;
        }

        .parchment-title {
            text-align: center;
            font-size: 1.2rem;
            font-weight: bold;
            text-transform: uppercase;
            border-bottom: 2px solid #2c1b0e;
            padding-bottom: 10px;
            margin-bottom: 15px;
            letter-spacing: 1px;
        }

        .parchment-content {
            flex-grow: 1;
            font-size: 1.1rem;
            font-style: italic;
            line-height: 1.6;
            margin-bottom: 15px;
        }

        .parchment-footer {
            font-size: 0.8rem;
            text-align: right;
            border-top: 1px dashed #2c1b0e;
            padding-top: 5px;
            opacity: 0.8;
            font-family: 'Segoe UI', sans-serif; /* Fonte legível para data */
            font-weight: bold;
        }

        /* Ícone de selo de cera (Decorativo) */
        .wax-seal {
            position: absolute;
            top: -10px;
            right: -10px;
            font-size: 2.5rem;
            text-shadow: 2px 2px 2px rgba(0,0,0,0.4);
        }

        /* --- ÁREA DE INPUT --- */
        .neon-textarea {
            width: 100%;
            height: 100px;
            background: rgba(0,0,0,0.6);
            border: 1px solid var(--primary);
            color: #fff;
            padding: 10px;
            border-radius: 5px;
            resize: vertical;
        }
        
        /* Responsividade */
        @media (max-width: 768px) {
            .parchment-section { grid-template-columns: 1fr; }
        }
    </style>
</head>
<body>
    <div class="light-rays"></div>

    <div class="container">
        
        <header class="header-game">
             <h1 style="color: var(--primary); font-family: 'Times New Roman', serif; text-transform: uppercase; letter-spacing: 2px; text-shadow: var(--text-neon-glow); font-size: 2rem; text-align: center; margin-bottom: 10px;">
                Ordens do Clã
             </h1>
             <p style="text-align: center; color: var(--neon-bright); margin-bottom: 15px;">
                 Clã: <?= htmlspecialchars($nome_cla) ?> | Seu Cargo: <?= $cargo_nome ?> (Nível <?= $cargo_id ?>)
             </p>
             
             <div class="btn-group navigation">
                 <a href="../../dashboardmed.php"><button style="width: auto;">Voltar ao Jogo</button></a>
             </div>
             <hr style="margin-top: 15px;">
        </header>

        <?php if (!empty($_SESSION['msg'])): ?>
            <div class="game-msg-box">
                <?= htmlspecialchars($_SESSION['msg']) ?>
            </div>
            <?php $_SESSION['msg'] = ''; ?>
        <?php endif; ?>

        <main class="page-grid">

            <div class="parchment-section">
                
                <div class="parchment-box">
                    <i class="fas fa-certificate wax-seal" style="color: #2e86de;"></i> <div class="parchment-title">Ordem Geral (OG)</div>
                    
                    <div class="parchment-content">
                        <?php if ($ultima_og): ?>
                            "<?= nl2br(htmlspecialchars($ultima_og['ordem'])) ?>"
                        <?php else: ?>
                            (Nenhuma Ordem Geral ativa no momento.)
                        <?php endif; ?>
                    </div>

                    <?php if ($ultima_og): 
                        $cargo_autor_og = $cargos_cla[$ultima_og['cargo']] ?? 'Membro do Clã';
                    ?>
                    <div class="parchment-footer">
                        Assinado: <?= htmlspecialchars($ultima_og['redator']) ?><br>
                        Cargo: <?= $cargo_autor_og ?><br>
                        Data: <?= date('d/m/Y H:i', strtotime($ultima_og['data'])) ?>
                    </div>
                    <?php endif; ?>
                </div>

                <div class="parchment-box">
                    <i class="fas fa-certificate wax-seal" style="color: #c0392b;"></i> <div class="parchment-title" style="color: #800000;">Ordem Maior (OM)</div>
                    
                    <div class="parchment-content">
                        <?php if ($ultima_om): ?>
                            <span style="color: #800000; font-weight: bold;">"<?= nl2br(htmlspecialchars($ultima_om['ordem'])) ?>"</span>
                        <?php else: ?>
                            (O Alto Comando está em silêncio.)
                        <?php endif; ?>
                    </div>

                    <?php if ($ultima_om): 
                         $cargo_autor_om = $cargos_cla[$ultima_om['cargo']] ?? 'Alto Comando';
                    ?>
                    <div class="parchment-footer" style="border-color: #800000; color: #800000;">
                        Decretado por: <?= htmlspecialchars($ultima_om['redator']) ?><br>
                        Cargo: <?= $cargo_autor_om ?><br>
                        Data: <?= date('d/m/Y H:i', strtotime($ultima_om['data'])) ?>
                    </div>
                    <?php endif; ?>
                </div>

            </div>

            <?php if ($pode_escrever_ordem): ?>
            <section class="game-panel" style="border-color: var(--primary);">
                <h2 class="panel-title">Redigir Novo Decreto</h2>
                <form action="" method="POST">
                    
                    <?php if ($pode_escolher_tipo): ?>
                    <div style="margin-bottom: 15px; display: flex; gap: 20px; color: #fff;">
                        <label style="cursor: pointer;">
                            <input type="radio" name="tipo" value="G" checked> 
                            <span style="color: var(--primary); font-weight: bold;">Ordem Geral</span>
                        </label>
                        <label style="cursor: pointer;">
                            <input type="radio" name="tipo" value="M"> 
                            <span style="color: var(--danger-color); font-weight: bold;">Ordem Maior</span>
                        </label>
                    </div>
                    <?php else: ?>
                        <input type="hidden" name="tipo" value="G">
                        <p style="color: #aaa; margin-bottom: 10px; font-size: 0.9rem;">Você está redigindo uma Ordem Geral.</p>
                    <?php endif; ?>

                    <textarea name="ordem" class="neon-textarea" placeholder="Escreva aqui a vontade do comando..."></textarea>
                    
                    <button type="submit" class="action-btn-neon" style="margin-top: 10px; width: 100%;">
                        <i class="fas fa-feather-alt"></i> Publicar Decreto
                    </button>
                </form>
            </section>
            <?php endif; ?>

            <?php if ($pode_ver_historico): ?>
            <section class="game-panel">
                <h2 class="panel-title">Arquivo Morto (Histórico)</h2>
                <div style="background: rgba(0,0,0,0.5); padding: 10px; border-radius: 5px; max-height: 300px; overflow-y: auto;">
                    <?php foreach ($historico as $h): 
                        $tipo_txt = ($h['tipo'] == 'M') ? 'OM' : 'OG';
                        $cor_txt = ($h['tipo'] == 'M') ? '#ff4444' : '#4444ff';
                    ?>
                        <div style="border-bottom: 1px solid #333; padding: 10px 0;">
                            <strong style="color: <?= $cor_txt ?>;">[<?= $tipo_txt ?>]</strong> 
                            <span style="color: #ccc;"><?= substr(htmlspecialchars($h['ordem']), 0, 60) ?>...</span>
                            <br>
                            <small style="color: #777;">
                                Por: <?= $h['redator'] ?> em <?= date('d/m H:i', strtotime($h['data'])) ?>
                            </small>
                        </div>
                    <?php endforeach; ?>
                    
                    <div style="text-align: center; margin-top: 10px;">
                        <a href="?p=<?= $pagina - 1 ?>" style="color: var(--primary); text-decoration: none;">&laquo; Anterior</a> | 
                        <a href="?p=<?= $pagina + 1 ?>" style="color: var(--primary); text-decoration: none;">Próximo &raquo;</a>
                    </div>
                </div>
            </section>
            <?php endif; ?>

        </main>
        
        <center>
        <footer class="footer-simple-medieval" style="margin-top: 30px;">
            <div class="footer-content">
                <p>&copy; 2025 RPG ERA MEDIEVAL</p>
            </div>
        </footer>
        </center>
    </div>
</body>
</html>