<?php
// criarpersonagem.php - CRIAÇÃO BLINDADA COM STATUS INICIAIS
ob_start();

// 1. CONEXÃO E SEGURANÇA CENTRALIZADA
require '../conexao.php';
require '../config_seguranca.php'; // Inicia sessão, previne XSS e Session Hijacking

// 2. VERIFICAÇÃO DE LOGIN
if (!isset($_SESSION['user_id'])) {
    header("Location: ../login.php");
    exit();
}

// Inicialização segura
$user_name = $_SESSION['user_name']; // Confiamos apenas na sessão
$csrf_token = $_SESSION['csrf_token'];
$successMessage = '';
$errorMessage = '';

// --- LÓGICA PHP (Processamento do POST) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['criar_personagem'])) {
    
    // 3. BLINDAGEM CSRF
    verificarTokenCSRF();

    try {
        $nome_personagem = trim($_POST['nome_personagem'] ?? '');
        $datahora_atual = date('Y-m-d H:i:s');

        // 4. VALIDAÇÕES RÍGIDAS
        if (strlen($nome_personagem) < 3 || strlen($nome_personagem) > 25) {
            throw new Exception("Nome deve ter entre 3 e 25 caracteres!");
        }

        // Permite apenas letras, números e espaços
        if (!preg_match('/^[a-zA-Z0-9À-ÿ ]+$/', $nome_personagem)) {
            throw new Exception("Caracteres inválidos no nome! Use apenas letras e números.");
        }

        // 5. TRANSAÇÃO ATÔMICA
        $pdo->beginTransaction();

        // A. Verifica nome duplicado (FOR UPDATE)
        $stmt = $pdo->prepare("SELECT id FROM personagens WHERE personagem = ? LIMIT 1 FOR UPDATE");
        $stmt->execute([$nome_personagem]);
        if ($stmt->fetch()) {
            throw new Exception("Este nome já está em uso! Escolha outro.");
        }

        // B. Verifica e Trava a Ficha (FOR UPDATE)
        $stmt = $pdo->prepare("SELECT id, hash FROM fichaspersonagens WHERE jogador = ? AND situacao = '1' LIMIT 1 FOR UPDATE");
        $stmt->execute([$user_name]);
        $ficha = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$ficha) {
            throw new Exception("Nenhuma ficha disponível! Adquira uma no mercado.");
        }

        // C. Executa a Queima da Ficha
        $stmt = $pdo->prepare("DELETE FROM fichaspersonagens WHERE id = ?");
        $stmt->execute([$ficha['id']]);

        // D. CRIA O PERSONAGEM COM STATUS INICIAIS (SUA SOLICITAÇÃO)
        // Define: Livre, Valerian, 10 Soldados, 200 Prata
        $stmt = $pdo->prepare("INSERT INTO personagens 
                               (personagem, jogador, datacriacao, situacao, local, soldados, prata) 
                               VALUES (?, ?, ?, 'livre', 'Cidade de Valerian', 10, 200)");
        
        $stmt->execute([$nome_personagem, $user_name, $datahora_atual]);
        $novo_personagem_id = $pdo->lastInsertId();

        // --- E. LOG NO INFOSSISTEMA ---
        
        // 1. Buscar Wallet do Criador
        $stmtW = $pdo->prepare("SELECT wallet FROM usersblock WHERE user = ? LIMIT 1");
        $stmtW->execute([$user_name]);
        $wallet_criador = $stmtW->fetchColumn();

        if (!$wallet_criador) $wallet_criador = "WALLET_NAO_ENCONTRADA";

        // 2. Montar Texto (Padrão Blockchain)
        $log_info = "CRIAÇÃO DE PERSONAGEM\n";
        $log_info .= "Wallet Criador: $wallet_criador\n";
        $log_info .= "Ficha Queimada: ID {$ficha['id']} (Hash: {$ficha['hash']})\n";
        $log_info .= "Personagem Gerado: ID $novo_personagem_id (Nome: $nome_personagem)\n";
        $log_info .= "Start: Valerian, 10 Soldados, 200 Prata";

        // 3. Inserir Log
        $stmtLog = $pdo->prepare("INSERT INTO infossistema (informacao, p1, p2, origem, dataehora) VALUES (?, ?, ?, 'Criação Personagem', ?)");
        $stmtLog->execute([$log_info, $wallet_criador, $novo_personagem_id, $datahora_atual]);

        // --- FIM DO LOG ---

        $pdo->commit();
        
        // Sucesso visual
        $nome_seguro = htmlspecialchars($nome_personagem);
        $_SESSION['creation_success'] = "Personagem <span class='highlight-neon-blue'>'$nome_seguro'</span> criado com sucesso!<br><small>Você recebeu: 10 Soldados e 200 Pratas.</small>";
        $_SESSION['last_creation'] = time();
        
        ob_end_clean();
        header("Location: ".$_SERVER['PHP_SELF']);
        exit();

    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        error_log("Erro Criação Personagem ($user_name): " . $e->getMessage());
        
        if (strpos($e->getMessage(), 'SQLSTATE') !== false) {
             $errorMessage = "Erro interno no sistema. Tente novamente.";
        } else {
             $errorMessage = $e->getMessage();
        }
    }
}

// Rate limiting visual
if (isset($_SESSION['last_creation']) && (time() - $_SESSION['last_creation'] < 5)) {
    // Apenas informativo
}

if (isset($_SESSION['creation_success'])) {
    $successMessage = $_SESSION['creation_success'];
    unset($_SESSION['creation_success']);
}

// Contagem Inicial (Visual)
$stmt = $pdo->prepare("SELECT COUNT(*) as total FROM fichaspersonagens WHERE jogador = ? AND situacao = '1'");
$stmt->execute([$user_name]);
$fichas_disponiveis = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
?>
<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Criar Personagem - Ferro & Fogo</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="../css/dashboard.css">
    
    <style>
        .highlight-neon-blue {
            color: #00f3ff;
            text-shadow: 0 0 5px #00f3ff;
            font-weight: bold;
        }
        /* Garantir input legível */
        input[type="text"] {
            background: rgba(0,0,0,0.7);
            border: 1px solid var(--primary);
            color: white;
            padding: 10px;
            width: 100%;
            box-sizing: border-box;
        }
    </style>
</head>
<body>
    <div class="light-rays"></div>

    <div class="container">
        
        <div class="card" style="margin-top: 30px; margin-bottom: 50px;">
            
            <div class="listing-header-centered">
                <h1 style="color: var(--primary); font-family: 'Times New Roman', serif; text-transform: uppercase; letter-spacing: 2px; text-shadow: var(--text-neon-glow); margin-bottom: 25px; font-size: 2rem;">
                    RPG ERA MEDIEVAL: Ferro & Fogo Pelo Poder
                </h1>
                
                <div class="btn-group btn-container-center" style="margin-bottom: 25px;">
   
                     <div class="btn-group nav-buttons-container">
                     <center>
                        <a href="../admpersonagens.php">
                            <button style="margin:0; padding: 12px 50px; width: auto;">Administração de Fichas & Personagens</button>
                        </a>
                        <br>
                        <a href="../dashboard.php">
                            <button style="margin:0; padding: 12px 50px; width: auto;">Menu Principal</button>
                        </a>
                        </center>
                    </div>
                    
                </div>

                <h2 style="color: white; text-transform: uppercase; letter-spacing: 1px; margin-bottom: 15px;">
                    Criar Personagem
                </h2>

                <p class="listing-intro">
                    Use uma de suas fichas para dar vida a um novo personagem no mundo de Ferro & Fogo.
                </p>
            </div>

            <?php if ($successMessage): ?>
                <div class="message-neon success" style="max-width: 700px; margin: 20px auto; border: 1px solid #00ff00; padding:15px; background:rgba(0,255,0,0.1); color:#00ff00; text-align:center;">
                    <?= $successMessage ?>
                </div>
            <?php endif; ?>
            
            <?php if ($errorMessage): ?>
                <div class="message-neon error" style="max-width: 700px; margin: 20px auto; border: 1px solid #ff0000; padding:15px; background:rgba(255,0,0,0.1); color:#ff0000; text-align:center;">
                    <?= htmlspecialchars($errorMessage) ?>
                </div>
            <?php endif; ?>

            <div class="fichas-counter" id="contador-fichas" style="text-align:center; font-size:1.2rem; margin: 20px 0;">
                Você possui <span class="highlight-neon" style="color:var(--primary); font-weight:bold; font-size:1.5rem;"><?= $fichas_disponiveis ?></span> ficha(s) disponível(is).
            </div>

            <div class="form-stack-container" style="max-width: 500px; margin: 0 auto;">
                <form method="POST" id="formPersonagem">
                    <input type="hidden" name="csrf_token" value="<?= $_SESSION['csrf_token'] ?>">
                    
                    <div>
                        <label for="nome_personagem" style="color:#ccc;">Nome do Personagem:</label>
                        <input type="text" id="nome_personagem" name="nome_personagem" required pattern="[a-zA-Z0-9À-ÿ ]{3,25}" placeholder="Digite o nome aqui..." autocomplete="off">
                        <small style="color:#777; display:block; margin-top:5px;">Mínimo 3 letras. Evite símbolos especiais.</small>
                    </div>
                    
                    <div class="btn-group" style="margin-top: 20px;">
                        <button type="submit" name="criar_personagem" onclick="return confirm('Tem certeza que deseja usar 1 ficha para criar este personagem?');">CRIAR PERSONAGEM</button>
                    </div>
                </form>
            </div>

        </div> 
        
        <footer class="footer-simple-medieval">
            <div class="footer-content">
                <h4>⚔️ RPG ERA MEDIEVAL: FERRO & FOGO ⚔️</h4>
                <p>&copy; <?php echo date("Y"); ?> Todos os direitos reservados.</p>
            </div>
        </footer>

    </div> 
    
    <script>
    // JS SEGURO: Não passa parâmetros na URL
    function atualizarFichas() {
        // Agora chamamos o arquivo SEM passar ?user=...
        // O PHP lá dentro vai saber quem somos pela SESSÃO
        fetch('atualizar_fichas.php')
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    const contador = document.getElementById('contador-fichas');
                    // Mantém o estilo, atualiza só o número e texto
                    contador.innerHTML = `Você possui <span class="highlight-neon" style="color:var(--primary); font-weight:bold; font-size:1.5rem;">${data.total}</span> ficha(s) disponível(is).`;
                } else if (data.redirect) {
                    // Se a sessão expirou, recarrega para ir pro login
                    window.location.reload();
                }
            })
            .catch(err => console.error('Erro silencioso:', err));
    }

    // Atualiza a cada 3 segundos
    setInterval(atualizarFichas, 3000);
    
    // Pequeno delay após submit para garantir atualização visual rápida
    document.getElementById('formPersonagem').addEventListener('submit', () => {
        setTimeout(atualizarFichas, 1000);
    });
    </script>
</body>
</html>