<?php
ob_start();
session_start();
require '../conexao.php';

// Verificação de login
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit();
}

// Gerenciamento de token CSRF
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Inicialização
$user_name = $_SESSION['user_name'];
$successMessage = '';
$errorMessage = '';

// Processamento do POST (ANTES de qualquer output)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['criar_personagem'])) {
    try {
        // Verificação CSRF
        if (!hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'] ?? '')) {
            throw new Exception("Token de segurança inválido!");
        }

        $nome_personagem = trim($_POST['nome_personagem'] ?? '');
        $datahora_atual = date('Y-m-d H:i:s');

        // Validações
        if (strlen($nome_personagem) < 3 || strlen($nome_personagem) > 25) {
            throw new Exception("Nome deve ter entre 3 e 25 caracteres!");
        }

        if (!preg_match('/^[a-zA-Z0-9À-ÿ ]+$/', $nome_personagem)) {
            throw new Exception("Caracteres inválidos no nome!");
        }

        $pdo->beginTransaction();

        // Verificação de nome duplicado COM LOCK
        $stmt = $pdo->prepare("SELECT id FROM personagens WHERE personagem = ? LIMIT 1 FOR UPDATE");
        $stmt->execute([$nome_personagem]);
        if ($stmt->fetch()) {
            throw new Exception("Nome já está em uso!");
        }

        // Obter ficha disponível COM LOCK
        $stmt = $pdo->prepare("SELECT id FROM fichaspersonagens WHERE jogador = ? AND situacao = '1' LIMIT 1 FOR UPDATE SKIP LOCKED");
        $stmt->execute([$user_name]);
        $ficha = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$ficha) {
            throw new Exception("Nenhuma ficha disponível encontrada!");
        }

        // Remover ficha usada
        $stmt = $pdo->prepare("DELETE FROM fichaspersonagens WHERE id = ?");
        $stmt->execute([$ficha['id']]);

        // Criar personagem
        $stmt = $pdo->prepare("INSERT INTO personagens 
            (personagem, jogador, datacriacao) 
            VALUES (?, ?, ?)");
        $stmt->execute([$nome_personagem, $user_name, $datahora_atual]);

        $pdo->commit();
        
        // Atualiza a sessão e redireciona
        $_SESSION['last_creation'] = time();
        $_SESSION['creation_success'] = "Personagem '$nome_personagem' criado com sucesso!";
        
        ob_end_clean();
        header("Location: ".$_SERVER['PHP_SELF']);
        exit();

    } catch (Exception $e) {
        $pdo->rollBack();
        $errorMessage = $e->getMessage();
    }
}

// Rate limiting (após possível redirecionamento)
if (isset($_SESSION['last_creation']) && (time() - $_SESSION['last_creation'] < 5)) {
    $errorMessage = "Aguarde 5 segundos entre criações.";
}

// Mensagens pós-redirecionamento
if (isset($_SESSION['creation_success'])) {
    $successMessage = $_SESSION['creation_success'];
    unset($_SESSION['creation_success']);
}

// Contar fichas ATUALIZADO (sincronizado com o banco)
$stmt = $pdo->prepare("SELECT COUNT(*) as total FROM fichaspersonagens WHERE jogador = ? AND situacao = '1'");
$stmt->execute([$user_name]);
$fichas_disponiveis = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
?>
<!DOCTYPE html>
<html>
<head>
    <title>Criar Personagem</title>
    <link rel="stylesheet" href="../style.css">
    <style>
        /* [Seus estilos CSS aqui] */
    </style>
</head>
<body>
<center>
    <h1>Criar Personagem</h1>
    
    <?php if ($successMessage): ?>
        <div class="success"><?= htmlspecialchars($successMessage) ?></div>
    <?php endif; ?>
    
    <?php if ($errorMessage): ?>
        <div class="error"><?= htmlspecialchars($errorMessage) ?></div>
    <?php endif; ?>

    <div id="contador-fichas">Você tem <?= $fichas_disponiveis ?> ficha(s)</div>

    <form method="POST" id="formPersonagem">
        <input type="hidden" name="csrf_token" value="<?= $_SESSION['csrf_token'] ?>">
        <input type="text" name="nome_personagem" required pattern="[a-zA-Z0-9À-ÿ ]{3,25}">
        <br><br>
        <button type="submit" name="criar_personagem">Criar</button>
    </form>

    <script>
    // Atualização em tempo real
    function atualizarFichas() {
        fetch('atualizar_fichas.php?user=<?= urlencode($user_name) ?>')
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    document.getElementById('contador-fichas').textContent = 
                        `Você tem ${data.total} ficha(s)`;
                }
            });
    }

    // Atualizar a cada 3 segundos e após submit
    setInterval(atualizarFichas, 3000);
    document.getElementById('formPersonagem').addEventListener('submit', atualizarFichas);
    </script>
</center>
</body>
</html>