<?php
// api/processar_jogo.php

// ATENÇÃO: Habilita log de erros temporariamente para debug
ini_set('display_errors', 0); 
error_reporting(E_ALL);
header('Content-Type: application/json; charset=utf-8');

function sendError($msg) {
    echo json_encode(['success' => false, 'message' => $msg]);
    exit;
}

try {
    // 1. VERIFICAÇÃO DE CAMINHOS (IGUAL AO QUE FUNCIONOU NO SALDO)
    if (file_exists('../conexao.php')) {
        require '../conexao.php';
    } elseif (file_exists('../../conexao.php')) {
        require '../../conexao.php';
    } else {
        throw new Exception("Arquivo 'conexao.php' não encontrado.");
    }
    
    if (file_exists('Financeiro.php')) {
        require 'Financeiro.php';
    } else {
        throw new Exception("Arquivo 'Financeiro.php' não encontrado na pasta api.");
    }

    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Método inválido.');
    }

    // 2. RECEBE DADOS
    $email = $_POST['email'] ?? '';
    $uid = $_POST['uid'] ?? '';
    $game_name = filter_input(INPUT_POST, 'game_name', FILTER_SANITIZE_SPECIAL_CHARS);
    $aposta = filter_input(INPUT_POST, 'aposta', FILTER_VALIDATE_FLOAT);
    $tempo = filter_input(INPUT_POST, 'duracao_por_minuto', FILTER_VALIDATE_INT);
    $limite = filter_input(INPUT_POST, 'limite_de_turnos_passados', FILTER_VALIDATE_INT);
    $mapSize = filter_input(INPUT_POST, 'mapSize', FILTER_VALIDATE_INT);
    
    // Validações
    if (!$email) throw new Exception('Usuário não identificado (Email vazio).');
    if (empty($game_name)) throw new Exception('Nome da missão obrigatório.');
    // Permite aposta 0, mas não negativa
    if ($aposta < 0) throw new Exception('Aposta inválida.');

    if (!$tempo) $tempo = 10;
    if (!$limite) $limite = 50;

    // 3. BUSCA USUÁRIO (USERSBLOCK)
    $stmt = $pdo->prepare("SELECT id, nome, saldo FROM usersblock WHERE email = ?");
    $stmt->execute([$email]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        // Se não existir, cria agora
        $stmtCreate = $pdo->prepare("INSERT INTO usersblock (email, saldo) VALUES (?, 0.00)");
        $stmtCreate->execute([$email]);
        $userId = $pdo->lastInsertId();
    } else {
        $userId = $user['id'];
        // Verificação extra de saldo antes de tentar transação
        if ($aposta > 0 && floatval($user['saldo']) < $aposta) {
            throw new Exception("Saldo insuficiente no servidor (R$ " . number_format($user['saldo'], 2) . ").");
        }
    }

    // 4. INICIA TRANSAÇÃO DO JOGO
    $banco = new Financeiro($pdo);
    
    // IMPORTANTE: O MySQL pode dar erro se tiver transação aberta pendente.
    // A classe Financeiro gerencia isso, mas vamos garantir aqui.
    if (!$pdo->inTransaction()) {
        $pdo->beginTransaction();
    }

    // 5. Verifica nome duplicado de jogo aberto
    // (Opcional, se tiver tabela games no MySQL)
    /*
    $stmtCheck = $pdo->prepare("SELECT COUNT(*) FROM games WHERE name = ? AND status = 'aberto'");
    $stmtCheck->execute([$game_name]);
    if ($stmtCheck->fetchColumn() > 0) {
        $pdo->rollBack();
        throw new Exception('Já existe uma operação com este nome.');
    }
    */

    // 6. Gera ID único
    $game_id_mysql = rand(100000, 999999);

    // 7. Desconta o dinheiro (Se houver aposta)
    if ($aposta > 0) {
        $resFin = $banco->processarTransacao(
            $userId, 
            -$aposta, // Negativo para descontar
            'APOSTA_JOGO', // <-- ISSO EXIGE O SQL DO PASSO 1
            "GAME-$game_id_mysql", 
            "Aposta: $game_name"
        );

        if (!$resFin['success']) {
            if ($pdo->inTransaction()) $pdo->rollBack();
            throw new Exception($resFin['msg']);
        }
    }

    // 8. Confirma
    if ($pdo->inTransaction()) $pdo->commit();

    echo json_encode([
        'success' => true, 
        'message' => 'Missão criada e aposta debitada!',
        'game_id' => $game_id_mysql,
        'novo_saldo' => $resFin['novo_saldo'] ?? 0 // Retorna o saldo novo se houve aposta
    ]);

} catch (Exception $e) {
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    // Retorna o erro exato para você ver no alerta
    sendError($e->getMessage());
}
?>