<?php
// Arquivo: api/Financeiro.php

class Financeiro {
    private $pdo;

    public function __construct($pdo) {
        if (!$pdo) {
            throw new Exception("Conexão com banco de dados inválida.");
        }
        $this->pdo = $pdo;
    }

    public function processarTransacao($userId, $valor, $tipo, $ref = null, $desc = "") {
        try {
            // Verifica se transação já está ativa para evitar erro de aninhamento
            $inTransaction = $this->pdo->inTransaction();
            if (!$inTransaction) {
                $this->pdo->beginTransaction();
            }

            // 1. Bloqueia a linha do usuário na tabela usersblock
            $stmt = $this->pdo->prepare("SELECT saldo FROM usersblock WHERE id = ? FOR UPDATE");
            $stmt->execute([$userId]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$user) {
                if (!$inTransaction) $this->pdo->rollBack();
                return ['success' => false, 'msg' => 'Usuário não encontrado na tabela usersblock (ID: ' . $userId . ').'];
            }

            $saldoAtual = floatval($user['saldo']);
            $novoSaldo = $saldoAtual + $valor;

            // 2. Validação de Saída (Não pode ficar negativo)
            if ($valor < 0 && $novoSaldo < 0) {
                if (!$inTransaction) $this->pdo->rollBack();
                return ['success' => false, 'msg' => 'Saldo insuficiente.'];
            }

            // 3. Registra Histórico na tabela transacoes
            $sqlHist = "INSERT INTO transacoes (user_id, tipo, valor, saldo_anterior, saldo_posterior, referencia_id, descricao, data_criacao) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, NOW())";
            $stmtHist = $this->pdo->prepare($sqlHist);
            $stmtHist->execute([$userId, $tipo, $valor, $saldoAtual, $novoSaldo, $ref, $desc]);

            // 4. Atualiza saldo final em usersblock
            $stmtUpd = $this->pdo->prepare("UPDATE usersblock SET saldo = ? WHERE id = ?");
            $stmtUpd->execute([$novoSaldo, $userId]);

            if (!$inTransaction) $this->pdo->commit();
            
            return ['success' => true, 'novo_saldo' => $novoSaldo, 'msg' => 'Transação aprovada.'];

        } catch (Exception $e) {
            if (!$inTransaction && $this->pdo->inTransaction()) {
                $this->pdo->rollBack();
            }
            error_log("Erro Financeiro: " . $e->getMessage());
            return ['success' => false, 'msg' => 'Erro interno: ' . $e->getMessage()];
        }
    }
}
?>