<?php
// api/entrar_jogo.php
// Responsável por debitar a aposta do P2 e confirmar a entrada

ini_set('display_errors', 0);
error_reporting(E_ALL);
header('Content-Type: application/json; charset=utf-8');

function sendError($msg) {
    echo json_encode(['success' => false, 'message' => $msg]);
    exit;
}

try {
    // 1. Dependências
    if (file_exists('../conexao.php')) require '../conexao.php';
    else require '../../conexao.php';
    
    if (file_exists('Financeiro.php')) require 'Financeiro.php';
    else throw new Exception("Arquivo Financeiro.php não encontrado.");

    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Método inválido.');
    }

    // 2. Dados
    $email = $_POST['email'] ?? '';
    // Recebe o game_id (MySQL ID)
    $game_id_mysql = filter_input(INPUT_POST, 'game_id', FILTER_VALIDATE_INT);
    $aposta = filter_input(INPUT_POST, 'aposta', FILTER_VALIDATE_FLOAT);

    if (!$email) throw new Exception('Usuário não identificado.');
    
    // Se tem aposta, precisa do ID do MySQL. Se aposta é 0, pode passar sem ID MySQL (jogos antigos/testes).
    if ($aposta > 0 && !$game_id_mysql) throw new Exception('ID do jogo inválido para aposta.');
    
    // 3. Busca Usuário (P2)
    $stmtUser = $pdo->prepare("SELECT id, saldo FROM usersblock WHERE email = ?");
    $stmtUser->execute([$email]);
    $user = $stmtUser->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        // Se não existe, cria (Sincronia)
        $stmtCreate = $pdo->prepare("INSERT INTO usersblock (email, saldo) VALUES (?, 0.00)");
        $stmtCreate->execute([$email]);
        $userId = $pdo->lastInsertId();
        $userSaldo = 0.00;
    } else {
        $userId = $user['id'];
        $userSaldo = floatval($user['saldo']);
    }

    // 4. Validação de Saldo Prévio
    if ($aposta > 0 && $userSaldo < $aposta) {
        throw new Exception("Saldo insuficiente (R$ " . number_format($userSaldo, 2) . ").");
    }

    // 5. TRANSAÇÃO GLOBAL
    $banco = new Financeiro($pdo);
    $pdo->beginTransaction();

    // 6. Debita a Aposta
    $novoSaldo = $userSaldo;
    if ($aposta > 0) {
        $resFin = $banco->processarTransacao(
            $userId, 
            -$aposta, 
            'APOSTA_JOGO', 
            "GAME-$game_id_mysql", 
            "Entrada na missão (P2)"
        );

        if (!$resFin['success']) {
            $pdo->rollBack();
            throw new Exception($resFin['msg']);
        }
        $novoSaldo = $resFin['novo_saldo'];
    }

    $pdo->commit();

    echo json_encode([
        'success' => true,
        'message' => 'Entrada confirmada! Dinheiro debitado.',
        'novo_saldo' => $novoSaldo
    ]);

} catch (Exception $e) {
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    sendError($e->getMessage());
}
?>